package com.xebialabs.deployit.plugin.satellite

import java.io.{ObjectInputStream, File}
import java.net.URI
import java.nio.file

import akka.actor.ActorSystem
import akka.pattern._
import akka.util.Timeout
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.engine.tasker.{TaskExecutionContext, TaskSpecification}
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.plugin.satellite.StreamingSupervisor.UploadRequest
import com.xebialabs.satellite.future.AwaitForever
import com.xebialabs.satellite.protocol.{Directories, Paths, ReplaceExtensionsFromTask, UploadReply}
import com.xebialabs.xlplatform.io.FolderChecksum
import com.xebialabs.xlplatform.satellite.Satellite

import scala.beans.BeanProperty
import scala.concurrent.Future
import scala.concurrent.duration._
import scala.util.{Failure, Success, Try}

case class SyncExtensionsStep(uploaderPath: String, actorLocator: ActorLocator, satelliteHostname: String, @BeanProperty description: String, xldInstalledPlugins: List[File], xldInstalledHotfixes: List[File], xldInstalledExt: List[File])
                             (implicit @transient var satelliteCommunicatorSystem: ActorSystem) extends Step with AwaitForever with SatellitePluginsSugar {

  override def execute(ctx: ExecutionContext): StepExitCode = {
    ctx.logOutput(s"Connecting to $actorLocator")

    val installedExtensions = askSatellitePlugin(ctx)

    (for {
      _ <- sendOrLog(installedExtensions.plugins, Directories.PLUGINS, ctx)
      _ <- sendOrLog(installedExtensions.hotfix, Directories.HOTFIXES, ctx)
      _ <- sendOrLog(installedExtensions.ext, Directories.EXT, ctx)
    } yield ()) match {
      case Success(_) => {
        if(List(installedExtensions.plugins, installedExtensions.hotfix, installedExtensions.ext).exists(_.nonEmpty)) {
          ctx.logOutput("Replacing extensions")
          ctx.setAttribute("restartRequired", true)
          implicit val timeout = Timeout(1.day)
          implicit val executionContext = satelliteCommunicatorSystem.dispatcher
          val task = ctx.getAttribute(TaskExecutionContext.CACHE_KEY).asInstanceOf[TaskSpecification]
          val satellitePluginManagement = actorLocator.locate(Paths.pluginManagement)
          blockOrThrow(satellitePluginManagement ? ReplaceExtensionsFromTask(task.getId))
        } else {
          ctx.logOutput("Replacing extensions not necessary")
        }
        StepExitCode.SUCCESS
      }
      case Failure(_) => StepExitCode.FAIL
    }

  }

  private def sendOrLog(ext: Set[URI], path: String, ctx: ExecutionContext) = {
    Try(if(ext.isEmpty) {
          ctx.logOutput(s"$path is up-to-date")
        } else {
          sendExtensions(ext, path, ctx)
        })
  }

  def sendExtensions(extensions: Set[URI], path: String, ctx: ExecutionContext) = {
    ctx.logOutput(s"Uploading $path to the satellite")

    val task = ctx.getAttribute(TaskExecutionContext.CACHE_KEY).asInstanceOf[TaskSpecification]

    val uploadRequests = extensions.map { pluginFileURI =>
      val pluginFile = PluginLocalFile(new File(pluginFileURI))

      val basePath = file.Paths.get(new java.io.File(".").getCanonicalPath)
      val filePath = file.Paths.get(pluginFile.getPath)
      val relativePath = basePath.relativize(filePath).getParent

      new UploadRequest(task.getId, relativePath.toString, pluginFile, ctx, actorLocator, satelliteHostname)
    }

    implicit val timeout = Timeout(1.day)
    implicit val executionContext = satelliteCommunicatorSystem.dispatcher

    val uploader = satelliteCommunicatorSystem.actorSelection(uploaderPath)

    val uploadResult = blockOrThrow(Future.sequence(uploadRequests.map(uploader ? _)))

    val errorInSendingFiles = uploadResult.collect {
      case UploadReply.Error(e) => e
    }

    if (errorInSendingFiles.nonEmpty) {
      ctx.logError(s"Error while sending $path to the satellite. (${errorInSendingFiles.mkString(", ")})")
      throw new Exception("error")
    }
  }

  @BeanProperty val order = Step.DEFAULT_ORDER

  private def readObject(in: ObjectInputStream): Unit = {
    in.defaultReadObject()
    satelliteCommunicatorSystem = SatelliteCommunicatorSystem.actorSystem
  }

}

object SyncExtensionsStep {

  def apply(satellite: Satellite)(implicit satelliteCommunicatorSystem: ActorSystem): SyncExtensionsStep = {
    new SyncExtensionsStep(
      SatelliteCommunicatorSystem.uploader.path.toSerializationFormat,
      ActorLocator(satellite), satellite.getAddress,
      s"Synchronise extensions on satellite ${satellite.getName}",
      FolderChecksum.listAllFilesSorted(new File(Directories.PLUGINS)),
      FolderChecksum.listAllFilesSorted(new File(Directories.HOTFIXES)),
      FolderChecksum.listAllFilesSorted(new File(Directories.EXT))
    )
  }

}
