package com.xebialabs.deployit.plugin.satellite

import java.io.File
import java.net.URI

import akka.actor.ActorSystem
import akka.pattern._
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.satellite.protocol.{Directories, Paths, GetInstalledExtensions, InstalledExtensions}
import com.xebialabs.xlplatform.io.FolderChecksum
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.concurrent.Await

case class InstalledExtensionsDelta(plugins: Set[URI], hotfix: Set[URI], ext: Set[URI])

trait SatellitePluginsSugar {

  def actorLocator: ActorLocator

  def xldInstalledPlugins: List[File]
  def xldInstalledHotfixes: List[File]
  def xldInstalledExt: List[File]

  def askSatellitePlugin(ctx: ExecutionContext)(implicit satelliteCommunicatorSystem: ActorSystem): InstalledExtensionsDelta = {
    implicit val timeout = CommonSettings(satelliteCommunicatorSystem).satellite.pingTimeout

    val satellitePluginManagement = actorLocator.locate(Paths.pluginManagement)

    val satelliteInstalledExtensions = Await.result((satellitePluginManagement ? GetInstalledExtensions).mapTo[InstalledExtensions], atMost = timeout.duration)

    InstalledExtensionsDelta(
      compareChecksum(satelliteInstalledExtensions.pluginsChecksum, xldInstalledPlugins),
      compareChecksum(satelliteInstalledExtensions.hotfixesChecksum, xldInstalledHotfixes),
      compareChecksum(satelliteInstalledExtensions.extChecksum, xldInstalledExt))
  }

  private def compareChecksum(checksum: String, files: List[File]): Set[URI] = {
    checksum == FolderChecksum.calculateChecksum(files) match {
      case false => files.map(_.toURI).toSet
      case true => Set.empty
    }
  }
}
