package com.xebialabs.deployit.plugin.satellite

import java.io.BufferedInputStream
import java.net.InetSocketAddress

import akka.actor._
import akka.stream.scaladsl._
import akka.stream.{ActorFlowMaterializer, ActorFlowMaterializerSettings}
import com.xebialabs.deployit.io.LazyLocalFile
import com.xebialabs.deployit.plugin.satellite.StreamingSupervisor.UploadRequest
import com.xebialabs.satellite.protocol.UploadReply.Ready
import com.xebialabs.satellite.protocol._
import com.xebialabs.satellite.streaming.DigesterStage.Digest
import com.xebialabs.satellite.streaming._
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{Await, Promise}
import scala.util.{Failure, Success, Try}

object FileUploader {
  def props(requester: ActorRef, request: UploadRequest): Props = Props(new FileUploader(requester, request))
}

class FileUploader(requester: ActorRef, request: UploadRequest) extends Actor with ActorLogging {

  val settings = ActorFlowMaterializerSettings(context.system)
  implicit val materializer = ActorFlowMaterializer(settings)
  implicit val system = context.system
  val securitySettings = CommonSettings(system).security

  var uploadIdleTimeout: FiniteDuration = _

  override def preStart() {
    val remoteUploadActor = request.satellite.locate(com.xebialabs.deployit.engine.tasker.satellite.Paths.tasks)

    remoteUploadActor ! UploadFileForTask(request.id, request.taskId, request.overthereFile.getName, request.path)
  }

  def receive = reachingSatellite

  def reachingSatellite: Receive = {
    case UploadReply.InitConnection(port, chunkSize, compression, wantTls) =>
      val satelliteAddress = new InetSocketAddress(request.satelliteHostname, port)
      log.info(s"'connecting to $satelliteAddress' for upload '${request.id}'")

      val fileStream = request.overthereFile.asInstanceOf[LazyLocalFile].getRawStream
      val buffStream = new BufferedInputStream(fileStream)

      val fileReceiver = sender()
      val satelliteReady = Promise[Unit]()

      implicit val streamingConfig = StreamConfig(
        chunkSize = chunkSize,
        compression = compression
      )

      val byteSource = UploadStage.source(satelliteReady.future, buffStream) {
        case Digest(checksum, fileSize) => self ! StreamDone(checksum, fileSize)
      }

      val connection = Tcp().outgoingConnection(satelliteAddress)

      Try {
        SslStreamingSupport.SslConfig(wantTls, securitySettings)
      } match {
        case Success(sslConfig) =>
          val wrappedConnection = SslStreamingSupport.wrapWithSsl(sslConfig.asClient.eagerClose, connection)
          val connectToDevNull = FlowGraph.closed(byteSource, wrappedConnection)((_, c) => c) {
            implicit builder => {
              (bs, wrappedConn) =>
                import FlowGraph.Implicits._
                bs ~> wrappedConn ~> Sink.ignore
            }
          }

          val runningFlow = connectToDevNull.run()

          import scala.concurrent.duration._
          val conn1 = Await.result(runningFlow, 2.seconds)

          log.info(s"Connected ${conn1.localAddress} -> ${conn1.remoteAddress}")
          fileReceiver ! Connected(conn1.localAddress)

          context become inSync(fileReceiver, satelliteReady)

        case Failure(exc) =>
          log.error(s"Cannot start connection: ${exc.getMessage}")
          log.debug(exc.toString)
          fileReceiver ! CannotConnect(exc.getMessage)
          requester ! UploadReply.Error(exc.getMessage)
          context stop self
      }
  }

  def inSync(fileReceiver: ActorRef, sync: Promise[Unit]): Receive = {
    case Ready =>
      sync.success(Unit)
      context become (streaming(fileReceiver) orElse handleError)

  }

  def streaming(fileReceiver: ActorRef): Receive = {
    case StreamDone(checksum, fileLength) =>
      fileReceiver ! FileUploaded(checksum, fileLength)

      request.ctx.logOutput(s"${request.path}/${request.overthereFile.getName}: uploaded $fileLength bytes. Checksum: $checksum")

    case UploadReply.Done =>
      log.info(s"'${request.path}' uploaded successfully")

      requester ! UploadReply.Done

      context stop self
  }

  def handleError: Receive = {
    case error@UploadReply.Error(cause) =>
      log.warning(s"'${request.overthereFile.getName}' failed")

      requester forward error

      context stop self
  }

  case class StreamDone(checksum: String, fileLength: Long)

}

