package com.xebialabs.deployit.plugin.satellite

import java.io.BufferedInputStream
import java.net.InetSocketAddress

import akka.actor._
import akka.stream.io.StreamTcp
import akka.stream.scaladsl._
import FlowGraphImplicits._
import akka.stream.{FlowMaterializer, MaterializerSettings}
import akka.util.ByteString
import com.xebialabs.deployit.io.LazyLocalFile
import com.xebialabs.deployit.plugin.satellite.StreamingSupervisor.UploadRequest
import com.xebialabs.satellite.protocol._
import com.xebialabs.satellite.streaming._
import com.xebialabs.xlplatform.settings.CommonSettings
import org.reactivestreams.Subscriber

import scala.concurrent.duration.FiniteDuration

abstract class FileUploader(requester: ActorRef, request: UploadRequest) extends Actor with ActorLogging {

  selfWith: ConnectionManager =>

  val settings = MaterializerSettings(context.system)
  implicit val materializer = FlowMaterializer(settings)
  implicit val system  = context.system

  var uploadIdleTimeout: FiniteDuration = _

  import request._

  override def preStart() {
    val remoteUploadActor = request.satellite.locate(com.xebialabs.deployit.engine.tasker.satellite.Paths.tasks)

    remoteUploadActor ! UploadFileForTask(id, request.taskId, request.overthereFile.getName, request.artifactId)

    uploadIdleTimeout = CommonSettings(context.system).satellite.uploadIdleTimeout.duration
    context.setReceiveTimeout(uploadIdleTimeout)
  }

  def receive = reachingSatellite orElse handleTimeout

  def handleTimeout: Receive = {
    case ReceiveTimeout =>

      requester ! UploadReply.Error(s"no connection to satellite within timeout ($uploadIdleTimeout)")

      context stop self
  }

  def reachingSatellite: Receive = {
    case UploadReply.InitConnection(port) =>
      val satelliteAddress = new InetSocketAddress(satelliteHostname, port)
      log.info(s"'$satelliteAddress' is ready for upload '$id'")

      IOManager ! StreamTcp.Connect(satelliteAddress)

      context become (waitingToBeConnected(sender()) orElse handleTimeout)
  }


  def waitingToBeConnected(fileReceiver: ActorRef, connection: Option[StreamTcp.OutgoingTcpConnection] = None): Receive = {
    case connection: StreamTcp.OutgoingTcpConnection =>
      fileReceiver ! Connected(connection.localAddress)

      log.info(s"Connected ${connection.localAddress} -> ${connection.remoteAddress}")

      context become (waitingForSatelliteToBeReady(fileReceiver, connection) orElse handleTimeout)
  }

  def waitingForSatelliteToBeReady(fileReceiver: ActorRef, connection: StreamTcp.OutgoingTcpConnection): Receive = {
    case UploadReply.Ready(chunkSize, compression) =>
      streamFile(connection.outputStream, connection.remoteAddress, fileReceiver, chunkSize, compression)
  }

  def streaming(fileReceiver: ActorRef): Receive = {
    case StreamDone(checksum, fileLength) =>
      fileReceiver ! FileUploaded(checksum, fileLength)

      request.ctx.logOutput(s"${request.artifactId}: uploaded $fileLength bytes. Checksum: $checksum")

    case UploadReply.Done =>
      log.info(s"'${request.artifactId}' uploaded successfully")

      requester ! UploadReply.Done

      context stop self

  }

  private def streamFile(outputStream: Subscriber[ByteString], satelliteAddress: InetSocketAddress, fileReceiver: ActorRef, chunkSize: Int, compression: Boolean) {
    request.ctx.logOutput(s"${request.artifactId}: streaming to $satelliteAddress")

    val fileStream = request.overthereFile.asInstanceOf[LazyLocalFile].getRawStream
    val buffStream = new BufferedInputStream(fileStream)
    val byteStream = new ByteIterator(buffStream)

    val streamDigester = new StreamDigester {
      override def onComplete() {
        self ! StreamDone(checksum, fileLength)

        buffStream.close()
      }
    }

    FlowGraph { implicit builder =>
      val broadcast = Broadcast[ByteString]
      val digester = SubscriberSink(streamDigester)

      val in = IteratorSource(byteStream.grouped(chunkSize).map(StreamOps.toByteString))

      val compress = if (compression) {
        Flow[ByteString].map(StreamOps.compress)
      } else {
        Flow[ByteString]
      }
      val out = SubscriberSink(outputStream)

      in ~> broadcast ~> compress ~> out
      broadcast ~> digester

    }.run()

    context become (streaming(fileReceiver) orElse handleError)

  }

  def handleError: Receive = {
    case error@UploadReply.Error(cause) =>
      log.warning(s"'${request.artifactId}' failed")

      requester forward error

      context stop self

  }

  case class StreamDone(checksum: String, fileLength: Long)

}

object FileUploader {

  def props(requester: ActorRef, request: UploadRequest): Props = Props(new FileUploader(requester, request) with StreamConnection)

}
