package com.xebialabs.deployit.plugin.satellite

import java.io.ObjectInputStream

import akka.actor.ActorSystem
import akka.util.Timeout
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.plugin.satellite.WaitingForRestartStep.{MaxAttemptReached, RetryOnceMore}
import com.xebialabs.satellite.protocol.Paths
import com.xebialabs.xlplatform.satellite.Satellite
import akka.pattern.ask
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.beans.BeanProperty
import scala.concurrent.duration._
import scala.concurrent.{Await, Future, Promise}
import scala.util.{Failure, Success}

class WaitingForRestartStep(maxAttempts: Int, delay: FiniteDuration, pingTimeout: FiniteDuration, actorLocator: ActorLocator)
                           (implicit @transient var satelliteCommunicatorSystem: ActorSystem) extends Step {

  @BeanProperty val order = Step.DEFAULT_ORDER
  @BeanProperty val description = "Waiting for XL-Satellite to be fully restarted."

  @transient implicit private lazy val dispatcher = satelliteCommunicatorSystem.dispatcher

  private var remainingAttempt: Int = maxAttempts

  override def execute(ctx: ExecutionContext): StepExitCode = {
    remainingAttempt = maxAttempts

    Await.result(satelliteRestarted(ctx), atMost = delay * maxAttempts * 2)
  }

  private def satelliteRestarted(ctx: ExecutionContext): Future[StepExitCode] = {
    remainingAttempt -= 1

    val result = Promise[StepExitCode]()

    if (remainingAttempt >= 0) {
      satelliteCommunicatorSystem.scheduler.scheduleOnce(delay) {

        implicit val timeout: Timeout = Timeout((pingTimeout * 1.5).asInstanceOf[FiniteDuration])

        ctx.logOutput(s"Connecting to $actorLocator")

        val pinger = satelliteCommunicatorSystem.actorOf(Pinger.props(ctx, pingTimeout, Clock()))

        (pinger ? Pinger.Start(actorLocator.locate(Paths.ping)))
          .mapTo[StepExitCode]
          .onComplete {
          case Success(exitCode) =>
            if (exitCode == StepExitCode.SUCCESS) {
              result.success(StepExitCode.SUCCESS)
              ctx.logOutput("Restart successful.")
            }
            else {
              result.failure(RetryOnceMore)
              ctx.logOutput("Retrying")
            }
          case Failure(_) =>
            result.failure(RetryOnceMore)
            ctx.logOutput("Retrying")
        }

      }
    } else {
      result.failure(MaxAttemptReached)
    }

    result.future.recoverWith {
      case RetryOnceMore => satelliteRestarted(ctx)
      case MaxAttemptReached => Future.successful(StepExitCode.FAIL)
    }
  }

  private def readObject(in: ObjectInputStream): Unit = {
    in.defaultReadObject()
    satelliteCommunicatorSystem = SatelliteCommunicatorSystem.actorSystem
  }
}

object WaitingForRestartStep {

  private object MaxAttemptReached extends Throwable

  private object RetryOnceMore extends Throwable

  def apply(satellite: Satellite, maxAttempts: Int, delay: Int)(implicit satelliteCommunicatorSystem: ActorSystem): WaitingForRestartStep =
    new WaitingForRestartStep(maxAttempts, delay.seconds, CommonSettings(satelliteCommunicatorSystem).satellite.remoteAskTimeout.duration, ActorLocator(satellite))
}