package com.xebialabs.deployit.plugin.satellite

import java.io.ObjectInputStream

import akka.actor.ActorSystem
import akka.util.Timeout
import akka.pattern.ask
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.satellite.protocol.Paths
import com.xebialabs.xlplatform.satellite.Satellite
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.beans.BeanProperty
import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{Await, Future}

case class PingSatelliteStep(actorLocator: ActorLocator, @BeanProperty description: String, clock: Clock = Clock())
                            (implicit @transient var satelliteCommunicatorSystem: ActorSystem) extends Step {

  def getOrder: Int = Step.DEFAULT_ORDER

  def execute(ctx: ExecutionContext): StepExitCode = {
    val pingTimeout = CommonSettings(satelliteCommunicatorSystem).satellite.pingTimeout.duration

    implicit val timeout: Timeout = Timeout((pingTimeout * 1.5).asInstanceOf[FiniteDuration])
    implicit val dispatcher = satelliteCommunicatorSystem.dispatcher

    ctx.logOutput(s"Connecting to $actorLocator")

    val pinger = satelliteCommunicatorSystem.actorOf(Pinger.props(ctx, pingTimeout, clock))

    val exitCode: Future[StepExitCode] = (pinger ? Pinger.Start(actorLocator.locate(Paths.ping)))
      .mapTo[StepExitCode]
      .recover { case _ =>
      ctx.logError(s"Operation timeout ($pingTimeout)")
      StepExitCode.FAIL
    }

    Await.result(exitCode, timeout.duration)
  }

  private def readObject(in: ObjectInputStream): Unit = {
    in.defaultReadObject()
    satelliteCommunicatorSystem = SatelliteCommunicatorSystem.actorSystem
  }
}

object PingSatelliteStep {

  def apply(satellite: Satellite)(implicit satelliteCommunicatorSystem: ActorSystem): PingSatelliteStep = {
    new PingSatelliteStep(ActorLocator(satellite), s"Checking connectivity with satellite ${satellite.getName}")
  }
}