package com.xebialabs.deployit.plugin.satellite

import java.io.File
import java.net.URI

import akka.pattern._
import akka.util.Timeout
import com.xebialabs.deployit.engine.tasker.satellite.SatellitePath
import com.xebialabs.deployit.engine.tasker.{TaskExecutionContext, TaskSpecification}
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.plugin.satellite.StreamingSupervisor.UploadRequest
import com.xebialabs.satellite.future.AwaitForever
import com.xebialabs.satellite.protocol.{Paths, ReplacePluginsFromTask, UploadReply}
import com.xebialabs.xlplatform.satellite.Satellite

import scala.beans.BeanProperty
import scala.concurrent.Future
import scala.concurrent.duration._

case class SyncPluginStep(uploaderPath: String, satellitePath: SatellitePath, satelliteHostname: String, @BeanProperty description: String, allPlugins: Set[URI]) extends Step with AwaitForever with SatellitePluginsSugar {

  override def execute(ctx: ExecutionContext): StepExitCode = {
    ctx.logOutput(s"Connecting to $satellitePath")

    val pluginsToUpload = askSatellitePlugin(ctx)

    if (pluginsToUpload.isEmpty) {
      ctx.logOutput(s"Satellite is up-to-date")
      StepExitCode.SUCCESS
    } else {
      ctx.logOutput(s"Uploading plugins to the satellite")
      sendPlugins(ctx)
    }
  }

  def sendPlugins(ctx: ExecutionContext): StepExitCode = {
    val task = ctx.getAttribute(TaskExecutionContext.CACHE_KEY).asInstanceOf[TaskSpecification]

    val uploadRequests = allPlugins.map { pluginFileURI =>
      val pluginFile = PluginLocalFile(new File(pluginFileURI))
      new UploadRequest(task.getId, pluginFile.getName, pluginFile, ctx, satellitePath, satelliteHostname)
    }

    implicit val timeout = Timeout(1.day)
    implicit val actorSystem = SatelliteActorSystem.actorSystem
    implicit val executionContext = actorSystem.dispatcher

    val uploader = SatelliteActorSystem.actorSystem.actorSelection(uploaderPath)

    val uploadResult = blockOrThrow(Future.sequence(uploadRequests.map(uploader ? _)))

    val errorInSendingFiles = uploadResult.collect {
      case UploadReply.Error(e) => e
    }

    if (errorInSendingFiles.nonEmpty) {
      ctx.logError(s"Error while sending plugin to the satellite. (${errorInSendingFiles.mkString(", ")})")
      StepExitCode.FAIL
    } else {
      ctx.logOutput("Replacing plugins")
      val satellitePluginManagement = satellitePath.locate(Paths.pluginManagement)
      blockOrThrow(satellitePluginManagement ? ReplacePluginsFromTask(task.getId))
      StepExitCode.SUCCESS
    }
  }

  @BeanProperty val order = Step.DEFAULT_ORDER

}

object SyncPluginStep {
  def apply(satellite: Satellite): SyncPluginStep = {
    new SyncPluginStep(
      SatelliteActorSystem.uploader.path.toSerializationFormat,
      SatellitePath(satellite), satellite.getAddress,
      s"Synchronizing plugins on satellites ${satellite.getName}", findAllXlDPlugins(new File("plugins")))
  }

  def findAllXlDPlugins(directory: File) = {
    val listFiles = Option(directory.listFiles())
    listFiles match {
      case Some(files) => files.filterNot(_.getName.endsWith("txt")).map(_.toURI).toSet
      case _ => Set.empty[URI]
    }
  }

}
