package com.xebialabs.deployit.plugin.satellite

import akka.pattern.ask
import com.xebialabs.deployit.engine.tasker.TaskId
import com.xebialabs.deployit.engine.tasker.satellite.SatellitePath
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.plugin.satellite.RestartSatelliteStep.RetryOnceMore
import com.xebialabs.satellite.future.AwaitForever
import com.xebialabs.satellite.protocol._
import com.xebialabs.xlplatform.satellite.Satellite

import scala.beans.BeanProperty
import scala.concurrent.duration._
import scala.concurrent.{Await, Future, Promise}
import scala.util.{Success, Try}


object RestartSatelliteStep {

  private object RetryOnceMore extends Throwable

  def apply(satellite: Satellite): RestartSatelliteStep = {
    new RestartSatelliteStep(
      satellitePath = SatellitePath(satellite), description = s"Checking tasks running on satellite ${satellite.getName}")
  }
}

case class RestartSatelliteStep(satellitePath: SatellitePath, @BeanProperty description: String) extends Step with AwaitForever {

  @BeanProperty val order = Step.DEFAULT_ORDER

  override def execute(ctx: ExecutionContext): StepExitCode = {
    executeTry(ctx).recover {
      case ex =>
        ctx.logError(s"Error: ${ex.getMessage}", ex)
        StepExitCode.FAIL
    }.get
  }

  private def executeTry(ctx: ExecutionContext): Try[StepExitCode] = Try {
    implicit val system = SatelliteActorSystem.actorSystem
    import system.dispatcher
    blockOn(satelliteStatus).get match {
      case SatelliteRunning(since) =>
        restartSatellite(ctx, Option(since))
      case Restarting =>
        Future.successful(StepExitCode.SUCCESS)
      case WaitingRestart(_) =>
        restartSatellite(ctx, None)
    }

    Await.result(restartSatellite(ctx), atMost = Duration.Inf)
  }

  private def satelliteStatus: Future[SatelliteStatus] = {
    implicit val satelliteTimeout = SatelliteActorSystem.config.remoteAskTimeout
    (locateRemoteActor ? GetSatelliteStatus).mapTo[SatelliteStatus]
  }

  def locateRemoteActor = {
    satellitePath.locate(Paths.satelliteControl)(SatelliteActorSystem.actorSystem)
  }

  private def restartSatellite(ctx: ExecutionContext, previousSatelliteRunningSince: Option[Long] = None) : Future[StepExitCode] = {

    implicit val satelliteTimeout = SatelliteActorSystem.config.remoteAskTimeout
    val result = Promise[StepExitCode]()
    implicit val system = SatelliteActorSystem.actorSystem
    import system.dispatcher

    def satelliteRestarting() {
      result.success(StepExitCode.SUCCESS)
      ctx.logOutput("Satellite is restarting")
    }

    def satelliteStillActive(runningTasks: Seq[TaskId]) {
      ctx.logOutput(s"Satellite is running some tasks ${runningTasks.mkString}")
      result.failure(RetryOnceMore)
    }

    def sendRestartMessage(): Unit = {
      ctx.logOutput(s"Resending Restart message")
      locateRemoteActor ! RestartSatellite
      result.failure(RetryOnceMore)
    }

    SatelliteActorSystem.scheduler.scheduleOnce(1.second) {
      satelliteStatus.onComplete {
        case Success(SatelliteRunning(since)) if Option(since) != previousSatelliteRunningSince =>
          satelliteRestarting()
        case Success(SatelliteRunning(since)) if Option(since) == previousSatelliteRunningSince =>
          sendRestartMessage()
        case Success(WaitingRestart(runningTasks)) =>
          satelliteStillActive(runningTasks)
        case Success(Restarting) =>
          satelliteRestarting()
        case _ =>
          ctx.logOutput("Could not contact Satellite, retrying...")
          result.failure(RetryOnceMore)
      }
    }

    result.future.recoverWith {
      case RetryOnceMore => restartSatellite(ctx, previousSatelliteRunningSince)
    }
  }
}
