package com.xebialabs.deployit.plugin.satellite

import akka.util.Timeout
import com.xebialabs.deployit.engine.tasker.satellite.SatellitePath
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.plugin.satellite.WaitingForRestartStep.{MaxAttemptReached, RetryOnceMore}
import com.xebialabs.satellite.protocol.Paths
import com.xebialabs.xlplatform.satellite.Satellite
import akka.pattern.ask

import scala.beans.BeanProperty
import scala.concurrent.duration._
import scala.concurrent.{Await, Future, Promise}
import scala.util.{Failure, Success}

class WaitingForRestartStep(
                             maxAttempts: Int,
                             delay: Int,
                             satellitePath: SatellitePath
                             ) extends Step {

  @BeanProperty val order = Step.DEFAULT_ORDER

  private var remainingAttempt: Int = maxAttempts

  private def intervalDuration = delay.seconds

  override def execute(ctx: ExecutionContext): StepExitCode = {
    remainingAttempt = maxAttempts

    Await.result(satelliteRestarted(ctx), atMost = intervalDuration * maxAttempts * 2)
  }

  private def satelliteRestarted(ctx: ExecutionContext): Future[StepExitCode] = {
    remainingAttempt -= 1
    implicit val system = SatelliteActorSystem.actorSystem
    val pingTimeout = SatelliteActorSystem.config.remoteAskTimeout.duration
    import system.dispatcher

    val result = Promise[StepExitCode]()

    if (remainingAttempt >= 0) {
      SatelliteActorSystem.scheduler.scheduleOnce(intervalDuration) {

        implicit val timeout: Timeout = Timeout((pingTimeout * 1.5).asInstanceOf[FiniteDuration])

        ctx.logOutput(s"Connecting to $satellitePath")

        val pinger = SatelliteActorSystem.actorSystem.actorOf(Pinger.props(ctx, pingTimeout, Clock()))

        (pinger ? Pinger.Start(satellitePath.locate(Paths.ping)))
          .mapTo[StepExitCode]
          .onComplete {
          case Success(exitCode) =>
            if (exitCode == StepExitCode.SUCCESS) {
              result.success(StepExitCode.SUCCESS)
              ctx.logOutput("Restart successful.")
            }
            else {
              result.failure(RetryOnceMore)
              ctx.logOutput("Retrying")
            }
          case Failure(_) =>
            result.failure(RetryOnceMore)
            ctx.logOutput("Retrying")
        }

      }
    } else {
      result.failure(MaxAttemptReached)
    }

    result.future.recoverWith {
      case RetryOnceMore => satelliteRestarted(ctx)
      case MaxAttemptReached => Future.successful(StepExitCode.FAIL)
    }
  }

  @BeanProperty val description = "Waiting for XL-Satellite to be fully restarted."
}

object WaitingForRestartStep {

  private object MaxAttemptReached extends Throwable

  private object RetryOnceMore extends Throwable

  def apply(satellite: Satellite, maxAttempts: Int, delay: Int): WaitingForRestartStep =
    new WaitingForRestartStep(maxAttempts, delay, SatellitePath(satellite))
}