package com.xebialabs.deployit.plugin.satellite

import akka.actor._
import akka.remote.{DisassociatedEvent, QuarantinedEvent}
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.io.ArtifactFile
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.satellite.UploadTaskSupervisor.Protocol.{ErrorWithAttemptCount, FileToUpload, UploadFilesTask, UploadTaskFinished}
import com.xebialabs.deployit.plugin.satellite.UploadTaskSupervisor.{MaxFileUploadAttempts, UploaderConstructor}
import com.xebialabs.satellite.protocol.{UploadId, UploadReply}
import com.xebialabs.xlplatform.settings.{CommonSettings, SatelliteSettings}

object UploadTaskSupervisor {
  type UploaderConstructor = (ActorRefFactory, ActorRef, FileToUpload, Int) => ActorRef

  object Protocol {

    case class UploadFilesTask(config: UploadConfig, files: Iterable[FileToUpload])

    case class UploadConfig(taskId: TaskId, ctx: ExecutionContext, satellite: ActorLocator, satelliteAddress: SatelliteAddress)

    case class FileToUpload(path: String, file: ArtifactFile, id: UploadId = UploadId())

    case class ErrorWithAttemptCount(cause: String, file: FileToUpload, uploadAttempt: Int)

    object EncryptionType {
      val ENCRYPTED = true
      val PLAIN = false
    }

    case class UploadTaskFinished(uploadedFiles: Long, error: Option[String] = None)

  }

  val MaxFileUploadAttempts = 20

  def props(requester: ActorRef, task: UploadFilesTask): Props = {
    val uploaderCons: UploaderConstructor = (factory, supervisor, file, uploadAttempt) =>
      factory.actorOf(FileUploader.props(supervisor, task.config, file, uploadAttempt), file.id.toString)
    props(requester, task, uploaderCons)
  }

  def props(requester: ActorRef, task: UploadFilesTask, uploaderCons: UploaderConstructor): Props = {
    Props(new UploadTaskSupervisor(requester, task, uploaderCons))
  }
}

class UploadTaskSupervisor(requester: ActorRef, task: UploadFilesTask, createUploader: UploaderConstructor)
  extends Actor with ActorLogging {
  val satelliteSettings: SatelliteSettings = CommonSettings(context.system).satellite

  val iterator: Iterator[FileToUpload] = task.files.iterator
  val levelOfParallelism: Int = satelliteSettings.maxConcurrentUploads
  val maxUploadAttempts: Int = if (satelliteSettings.fileUploadAttempts > MaxFileUploadAttempts) MaxFileUploadAttempts else satelliteSettings.fileUploadAttempts
  var concurrentUploads = 0
  var uploadedFiles = 0

  @throws[Exception](classOf[Exception])
  override def preStart(): Unit = {
    context.system.eventStream.subscribe(self, classOf[DisassociatedEvent])
    context.system.eventStream.subscribe(self, classOf[QuarantinedEvent])

    if (task.files.isEmpty) {
      finishedSuccessfully()
    } else {
      log.info(s"Starting files upload with maximum $levelOfParallelism files in parallel")
      0.to(levelOfParallelism).foreach(_ => uploadNextFile())
    }
  }

  override def receive: Receive = {
    case ErrorWithAttemptCount(cause, fileToUpload, uploadAttempts) =>
      log.info(s"Error occured during file upload due to $cause")
      if (uploadAttempts < maxUploadAttempts) {
        log.info(s"Retry $uploadAttempts times file upload of ${fileToUpload.path}")
        createUploader(context, self, fileToUpload.copy(id = UploadId()), uploadAttempts + 1)
      } else {
        requester ! UploadTaskFinished(uploadedFiles, Some(cause))
        log.info(s"Stopping file uploading actor now!!")
        context.stop(self)
      }

    case UploadReply.Error(cause) =>
      log.info(s"Error occured during file upload due to $cause")
      requester ! UploadTaskFinished(uploadedFiles, Some(cause))
      log.info(s"Stopping file uploading actor now!!")
      context.stop(self)

    case UploadReply.Done =>
      concurrentUploads -= 1
      uploadedFiles += 1
      log.info(s"Successfully uploaded file number $uploadedFiles, going to upload next file now")
      uploadNextFile()

    case _: DisassociatedEvent =>
      requester ! UploadTaskFinished(uploadedFiles, Some("The satellite seems to be unreachable at the moment. Please try again later."))
      context.stop(self)

    case _: QuarantinedEvent =>
      requester ! UploadTaskFinished(uploadedFiles, Some("The satellite seems to be permanently unreachable. Please restart the satellite and retry."))
      context.stop(self)
  }

  private def uploadNextFile(): Unit = {
    if (iterator.hasNext && concurrentUploads < levelOfParallelism) {
      val file = iterator.next()
      log.info(s"fetched next file ${file.path} from the list, going to upload it now..")
      val firstAttempt = 1
      createUploader(context, self, file, firstAttempt)
      concurrentUploads += 1
    }
    if (concurrentUploads == 0) {
      log.info(s"Done uploading $uploadedFiles files, no more files left to upload!!")
      finishedSuccessfully()
    }
  }

  private def finishedSuccessfully(): Unit = {
    log.info("Stopping the upload file actor now!!")
    requester ! UploadTaskFinished(uploadedFiles)
    context.stop(self)
  }
}
