package com.xebialabs.satellite.serialization

import com.esotericsoftware.kryo.Kryo
import com.esotericsoftware.kryo.io.{Input, Output}
import com.esotericsoftware.kryo.serializers.FieldSerializer.CachedField
import com.google.common.io
import com.xebialabs.deployit.io.{DerivedArtifactFile, Exploder}
import com.xebialabs.deployit.plugin.api.udm.artifact.{Artifact, DerivedArtifact, FolderArtifact, SourceArtifact}
import com.xebialabs.overthere.local.LocalFile

import scala.util.Try


class ArtifactSerializer(kryo: Kryo, val typeClass: Class[_ <: Artifact], data: Either[FilesToUploadCache, UploadedFiles]) extends BaseCustomSerializer[Any](kryo, typeClass) {

  override def initializeCachedFields() {

    val fileFields = fetchFileFields(typeClass)

    val extractExistingFileFields = fileFields.filter(fileField => isFieldExistsOnClass(fileField))

    extractExistingFileFields.map(field => getField(field.getName)).foreach(removedFields.add)
  }

  override def removeField(fieldName: ArtifactId): Unit = Try(super.removeField(fieldName))

  override def removeField(removeField: CachedField[_]): Unit = Try(super.removeField(removeField))

  override def read(kryo: Kryo, input: Input, `type`: Class[Any]): Any = data match {

    case Right(files) =>
      val customRead = super.read(kryo, input, `type`)

      customRead match {
        case value: SourceArtifact =>
          val artifactId = value.getId

          val uploadedFile = files(artifactId)

          if (value.isInstanceOf[FolderArtifact]) {
            Exploder.explode(io.Files.asByteSource(uploadedFile), uploadedFile.getParentFile, null)
            value.setFile(LocalFile.valueOf(uploadedFile.getParentFile))
          } else {
            value.setFile(LocalFile.valueOf(uploadedFile))
          }
        case value: DerivedArtifact[_] =>
          value.setFile(DerivedArtifactFile.create(value))
        case _ =>
      }

      customRead

    case _ => super.read(kryo, input, `type`)
  }


  override def write(kryo: Kryo, output: Output, obj: Any) = data match {
    case Left(fileMap) => obj match {
      case srcArtifact: SourceArtifact =>
        fileMap += srcArtifact
      case _ =>
    }

      super.write(kryo, output, obj)

    case _ => super.write(kryo, output, obj)
  }

}