package com.xebialabs.xlplatform.test.ci

import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.repository.{JcrRepositoryService, RepositoryService}
import com.xebialabs.overthere.OverthereFile

import scala.language.implicitConversions

object CiHelper {
  implicit def stringToType(t: String): Type = Type.valueOf(t)
  implicit def classToType[T <: ConfigurationItem](c: Class[T]): Type = Type.valueOf(c)
  implicit def overhereFileToSome(f: OverthereFile): Option[OverthereFile] = Option(f)
}

class CiHelper(repositoryService: RepositoryService) {
  import com.xebialabs.xlplatform.test.ci.CiHelper._

  repositoryService match {
    case r: JcrRepositoryService => r.setLicenseService(new NoLicenseService)
    case _ => // do nothing
  }

  def constructCi[T <: ConfigurationItem](t: Type, id: String): T = {
    DescriptorRegistry.getDescriptor(t).newInstance(id)
  }

  def createCi[T <: ConfigurationItem](t: Type, id: String): T = {
    val ci: T = constructCi(t, id)
    repositoryService.create(ci)
    ci
  }

  def constructConfigurationItem[T <: ConfigurationItem](clazz: Class[T], id: String): T = constructCi(clazz, id)

  def createConfigurationItem[T <: ConfigurationItem](clazz: Class[T], id: String): T = createCi(clazz, id)

  def createDirectory(id: String): Directory = createConfigurationItem(classOf[Directory], id)


  def constructArtifact[T <: Artifact](clazz: Class[T], id: String, file: OverthereFile): T = {
    val ci = constructConfigurationItem(clazz, id)
    if (file != null)
      ci.setFile(file)
    ci
  }

  def createArtifact[T <: Artifact](clazz: Class[T], id: String, file: OverthereFile): T = {
    val a = constructArtifact(clazz, id, file)
    repositoryService.create(a)
    a
  }

  def createApplication(id: String): Application = createConfigurationItem(classOf[Application], "Applications/" + id)

  def constructPackage(id: String, app: Application, members: Deployable*): DeploymentPackage = {
    val pkg = constructConfigurationItem(classOf[DeploymentPackage], app.getId + "/" + id)
    pkg.setApplication(app)
    pkg.setDeployables(newHashSet(members.toSet))
    pkg
  }

  def createPackage(id: String, app: Application, members: Deployable*): DeploymentPackage = {
    val pkg = constructPackage(id, app, members: _*)
    repositoryService.create(pkg)
    pkg
  }

  def createComposite(id: String, app: Application, packages: Version*): CompositePackage = {
    val composite = constructConfigurationItem(classOf[CompositePackage], app.getId + "/" + id)
    composite.setApplication(app)
    composite.setProperty("packages", newArrayList(packages))
    repositoryService.create(composite)
    composite
  }

  def createContainer[C <: Container](id: String, clazz: Class[C]): C = createConfigurationItem(clazz, "Infrastructure/" + id)

  def createEnvironment(id: String, members: Container*): Environment = {
    val env = constructConfigurationItem(classOf[Environment], "Environments/" + id)
    env.setMembers(newHashSet(members.toSet))
    repositoryService.create(env)
    env
  }

  def constructDeployed[D <: Deployable, C <: Container, DC <: Deployed[D, C]](clazz: Class[DC], deployable: D, container: C): DC = {
    val deployed = constructConfigurationItem(clazz, container.getId + "/" + deployable.getId.split("/").last)
    deployed.setDeployable(deployable)
    deployed.setContainer(container)
    deployed
  }

  def constructDeployment(version: Version, env: Environment,
                          deployeds: Deployed[_ <: Deployable, _ <: Container]*): DeployedApplication = {
    val deployment = constructConfigurationItem(classOf[DeployedApplication],
      env.getId + "/" + version.getApplication.getId.split("/").last)
    deployment.setVersion(version)
    deployment.setEnvironment(env)
    deployment.setDeployeds(newHashSet(deployeds.toSet))
    deployment
  }

  def createDeployment(version: Version, env: Environment,
                       deployeds: Deployed[_ <: Deployable, _ <: Container]*): DeployedApplication = {
    val d = constructDeployment(version, env, deployeds: _*)
    repositoryService.create(d)
    d
  }

  // Define several shortcuts for methods with variable arguments to make life easier for Java clients of this helper

  def createPackage(id: String, app: Application): DeploymentPackage = {
    createPackage(id, app, Seq(): _*)
  }
  def createPackage(id: String, app: Application, member: Deployable): DeploymentPackage = {
    createPackage(id, app, Seq(member): _*)
  }
  def createPackage(id: String, app: Application, member1: Deployable, member2: Deployable): DeploymentPackage = {
    createPackage(id, app, Seq(member1, member2): _*)
  }

  def createEnvironment(id: String): Environment = {
    createEnvironment(id, Seq(): _*)
  }
  def createEnvironment(id: String, member: Container): Environment = {
    createEnvironment(id, Seq(member): _*)
  }
  def createEnvironment(id: String, member1: Container, member2: Container): Environment = {
    createEnvironment(id, Seq(member1, member2): _*)
  }

  def constructDeployment(version: Version, env: Environment): DeployedApplication = {
    constructDeployment(version, env, Seq(): _*)
  }
  def constructDeployment(version: Version, env: Environment,
                          deployed: Deployed[_ <: Deployable, _ <: Container]): DeployedApplication = {
    constructDeployment(version, env, Seq(deployed): _*)
  }

  def createDeployment(version: Version, env: Environment): DeployedApplication = {
    createDeployment(version, env, Seq(): _*)
  }

  // Make sure that converted maps are serializable
  private def newHashSet[T](s: Set[T]) = new java.util.HashSet[T](scala.collection.JavaConversions.setAsJavaSet(s))
  private def newArrayList[T](s: Seq[T]) = new java.util.ArrayList[T](scala.collection.JavaConversions.seqAsJavaList(s))

}
