package com.xebialabs.xlplatform.extensions.maven

import com.typesafe.config.Config
import com.xebialabs.deployit.repository.WorkDirFactory
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlplatform.config.{ConfigLoader, ConfigurationHolder}
import org.eclipse.aether.repository.RepositoryPolicy.{CHECKSUM_POLICY_WARN, UPDATE_POLICY_DAILY}
import org.eclipse.aether.repository.{Authentication, Proxy, RemoteRepository, RepositoryPolicy}
import org.eclipse.aether.util.repository.AuthenticationBuilder

import scala.jdk.CollectionConverters._

object MavenSettings {
  def apply(config: Config) = new MavenSettings(config)

  def apply(): MavenSettings = MavenSettings(defaultConfig)

  lazy val defaultConfig: Config = ConfigLoader.loadWithDynamic(ConfigurationHolder.get())
}

class MavenSettings(val config: Config) {

  lazy val repositories: List[RemoteRepository] = readRepositories()

  lazy val workDirFactory = new WorkDirFactory(config.getString("xl.artifact.resolver.maven.work-dir"))

  lazy val ignoreMissingArtifact: Boolean = config.getBoolean("xl.artifact.resolver.maven.ignoreMissingArtifact")

  lazy val passwordEncrypter: PasswordEncrypter = PasswordEncrypter.getInstance()

  private[maven] def readRepositories(): List[RemoteRepository] = {
    try {
      val repositories = config.getConfigList("xl.artifact.resolver.maven.repositories").asScala.map { repoConfig =>
        val builder = new RemoteRepository.Builder(repoConfig.getString("id"), "default", repoConfig.getString("url"))

        builder.setAuthentication(readAuthentication(repoConfig).orNull)
        builder.setProxy(readProxy(repoConfig).orNull)

        builder.setReleasePolicy(readPolicy(repoConfig, "releases"))
        builder.setSnapshotPolicy(readPolicy(repoConfig, "snapshots"))

        builder.build()
      }.toList
      repositories.groupBy(_.getId).collect {
        case (x, List(_, _, _*)) =>
          throw new IllegalArgumentException(s"There is duplicate repository ID in Maven configuration: $x")
      }
      repositories
    } catch {
      case ex: Exception =>
        throw new IllegalStateException(s"Could not parse Maven repositories configuration: ${ex.getMessage}", ex)
    }
  }

  private def readAuthentication(repoConfig: Config): Option[Authentication] = {
    repoConfig.getConfigOption("authentication").map { auth =>
      new AuthenticationBuilder()
        .addUsername(auth.getStringOrNull("username"))
        .addPassword(passwordEncrypter.ensureDecrypted(auth.getStringOrNull("password")))
        .addPrivateKey(auth.getStringOrNull("privateKey"), auth.getStringOrNull("passphrase"))
        .build()
    }
  }

  private def readProxy(repoConfig: Config): Option[Proxy] = {
    repoConfig.getConfigOption("proxy").map { proxyConfig =>
      val protocol = if (repoConfig.getString("url").toLowerCase.startsWith("https:"))
        Proxy.TYPE_HTTPS
      else Proxy.TYPE_HTTP
      val auth = proxyConfig.getStringOption("username").map(
        new AuthenticationBuilder().addUsername(_).addPassword(passwordEncrypter.ensureDecrypted(proxyConfig.getString("password"))).build()
      ).orNull
      new Proxy(protocol, proxyConfig.getString("host"), proxyConfig.getInt("port"), auth)
    }
  }

  private def readPolicy(repoConfig: Config, policyName: String): RepositoryPolicy = {
    repoConfig.getConfigOption(policyName) match {
      case Some(policyConfig) =>
        new RepositoryPolicy(
          policyConfig.getBooleanOption("enabled").getOrElse(true),
          policyConfig.getStringOption("updatePolicy").getOrElse(UPDATE_POLICY_DAILY),
          policyConfig.getStringOption("checksumPolicy").getOrElse(CHECKSUM_POLICY_WARN)
        )
      case None => new RepositoryPolicy()
    }
  }

  private implicit class ConfigWithOptions(config: Config) {

    def getConfigOption: String => Option[Config] = getValueOption(config.getConfig)

    def getStringOption: String => Option[String] = getValueOption(config.getString)

    def getBooleanOption: String => Option[Boolean] = getValueOption(config.getBoolean)

    def getStringOrNull(path: String): String = getStringOption(path).orNull

    private def getValueOption[T](getter: String => T)(path: String): Option[T] =
      if (config.hasPath(path)) {
        Some(getter(path))
      } else {
        None
      }
  }

}
