package com.xebialabs.xlplatform.extensions.maven

import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.repository.WorkDirFactory
import org.eclipse.aether.repository.RepositoryPolicy.{CHECKSUM_POLICY_WARN, UPDATE_POLICY_DAILY}
import org.eclipse.aether.repository.{Authentication, Proxy, RemoteRepository, RepositoryPolicy}
import org.eclipse.aether.util.repository.AuthenticationBuilder

import scala.collection.convert.ImplicitConversions._

object MavenSettings {
  def apply(config: Config) = new MavenSettings(config)

  def apply(): MavenSettings = MavenSettings(defaultConfig)

  lazy val defaultConfig: Config = ConfigFactory.defaultOverrides()
    .withFallback(ConfigFactory.parseResources("maven.conf"))
    .withFallback(ConfigFactory.parseMap(ServerConfiguration.getInstance().getCustomPasswords))
    .withFallback(ConfigFactory.defaultReference())
    .resolve()
}

class MavenSettings(val config: Config) {

  lazy val repositories = readRepositories()

  lazy val workDirFactory = new WorkDirFactory(config.getString("maven.work-dir"))

  lazy val ignoreMissingArtifact: Boolean = config.getBoolean("xl.repository.artifact.resolver.maven.ignoreMissingArtifact")

  private[maven] def readRepositories(): List[RemoteRepository] = {
    try {
      val repositories = config.getConfigList("maven.repositories").map { repoConfig =>
        val builder = new RemoteRepository.Builder(repoConfig.getString("id"), "default", repoConfig.getString("url"))

        builder.setAuthentication(readAuthentication(repoConfig).orNull)
        builder.setProxy(readProxy(repoConfig).orNull)

        builder.setReleasePolicy(readPolicy(repoConfig, "releases"))
        builder.setSnapshotPolicy(readPolicy(repoConfig, "snapshots"))

        builder.build()
      }.toList
      repositories.groupBy(_.getId).collect {
        case (x, List(_, _, _*)) =>
          throw new IllegalArgumentException(s"There is duplicate repository ID in Maven configuration: $x")
      }
      repositories
    } catch {
      case ex: Exception =>
        throw new IllegalStateException(s"Could not parse Maven repositories configuration: ${ex.getMessage}", ex)
    }
  }

  private def readAuthentication(repoConfig: Config): Option[Authentication] = {
    repoConfig.getConfigOption("authentication").map { auth =>
      new AuthenticationBuilder()
        .addUsername(auth.getStringOrNull("username"))
        .addPassword(auth.getStringOrNull("password"))
        .addPrivateKey(auth.getStringOrNull("privateKey"), auth.getStringOrNull("passphrase"))
        .build()
    }
  }

  private def readProxy(repoConfig: Config): Option[Proxy] = {
    repoConfig.getConfigOption("proxy").map { proxyConfig =>
      val protocol = if (repoConfig.getString("url").toLowerCase.startsWith("https:"))
        Proxy.TYPE_HTTPS
      else Proxy.TYPE_HTTP
      val auth = proxyConfig.getStringOption("username").map(
        new AuthenticationBuilder().addUsername(_).addPassword(proxyConfig.getString("password")).build()
      ).orNull
      new Proxy(protocol, proxyConfig.getString("host"), proxyConfig.getInt("port"), auth)
    }
  }

  private def readPolicy(repoConfig: Config, policyName: String): RepositoryPolicy = {
    repoConfig.getConfigOption(policyName) match {
      case Some(policyConfig) =>
        new RepositoryPolicy(
          policyConfig.getBooleanOption("enabled").getOrElse(true),
          policyConfig.getStringOption("updatePolicy").getOrElse(UPDATE_POLICY_DAILY),
          policyConfig.getStringOption("checksumPolicy").getOrElse(CHECKSUM_POLICY_WARN)
        )
      case None => new RepositoryPolicy()
    }
  }

  private implicit class ConfigWithOptions(config: Config) {

    def getConfigOption = getValueOption(config.getConfig) _

    def getStringOption = getValueOption(config.getString) _

    def getBooleanOption = getValueOption(config.getBoolean) _

    def getStringOrNull(path: String): String = getStringOption(path).orNull

    private def getValueOption[T](getter: String => T)(path: String): Option[T] = {
      config.hasPath(path) match {
        case true => Some(getter(path))
        case false => None
      }
    }
  }

}
