package com.xebialabs.xlplatform.extensions.exportcis.archive

import com.thoughtworks.xstream.io.xml.JDom2Reader
import com.xebialabs.deployit.core.xml.PasswordEncryptingCiConverter
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.repository._
import com.xebialabs.deployit.repository.internal.Root
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xlplatform.artifact.ExternalArtifactDownloader.downloadResolvedFile
import com.xebialabs.xlplatform.artifact.resolution.ArtifactResolverRegistry.{isExternalArtifact, isStoredArtifact}
import com.xebialabs.xlplatform.extensions.exportcis.archive.RepositoryExporter._
import com.xebialabs.xlplatform.extensions.exportcis.archive.RepositoryImporter._
import com.xebialabs.xlplatform.extensions.exportcis.archive.RepositoryXmlExporter.Attributes._
import com.xebialabs.xlplatform.extensions.exportcis.archive.RepositoryXmlExporter.Elements._
import com.xebialabs.xlplatform.extensions.exportcis.archive.RepositoryXmlExporter._
import com.xebialabs.xlplatform.sugar.PathSugar._
import com.xebialabs.xlplatform.utils.ResourceManagement._
import com.xebialabs.xltype.serialization.CiReader
import com.xebialabs.xltype.serialization.xstream.CiXstreamReader
import grizzled.slf4j.Logging
import org.jdom2.input.SAXBuilder
import org.jdom2.{Element, Namespace}

import java.io.{File, FileOutputStream}
import java.nio.file.Paths
import java.security.InvalidKeyException
import java.util.zip.ZipFile
import scala.jdk.CollectionConverters._
import scala.util.Try


object RepositoryImporter {

  implicit class RepositoryWithCleanup(val repositoryService: RepositoryService) {

    def deleteBeforeImport(path: String): Unit = path match {
      case "" | RepositoryRootAlias => deleteContentsOf(RepositoryRootAlias)
      case p if isInternalRoot(repositoryService.read[ConfigurationItem](path, 0).getType) =>
        deleteContentsOf(p)
      case p =>
        repositoryService.delete(p)
    }

    private def isInternalRoot(t: Type): Boolean = t.instanceOf(Type.valueOf(classOf[Root]))

    private def deleteContentsOf(path: String): Unit = {
      val nonRoots = repositoryService.list(new SearchParameters().setParent(path)).asScala.filter {
        case ciData if isInternalRoot(ciData.getType) =>
          deleteContentsOf(ciData.getId)
          false
        case _ => true
      }

      repositoryService.delete(nonRoots.map(_.getId).toSeq: _*)
    }

  }

}

class RepositoryImporter(val repositoryService: RepositoryService, val ctx: ExecutionContext) extends Logging {

  private val SHA1_LENGTH = 40
  private val SHA256_LENGTH = 64

  def `import`(archivePath: File): Try[Unit] = Try {

    implicit val namespace: Namespace = XlNamespace
    using(new ZipFile(archivePath)) { zipArchive =>
      using(zipArchive.entryStream(CiXmlFileName)) { cisXmlIs =>

        val saxBuilder: SAXBuilder = new SAXBuilder
        val rootElement: Element = saxBuilder.build(cisXmlIs).getRootElement

        val archiveFingerprint = rootElement.child(metadata, encryptionKeyFingerprint).getValue
        val currentFingerprint = archiveFingerprint match {
          case fp if fp.length == SHA1_LENGTH => PasswordEncrypter.getInstance().getKeyFingerprint("SHA-1")
          case fp if fp.length == SHA256_LENGTH => PasswordEncrypter.getInstance().getKeyFingerprint("SHA-256")
          case _ =>
            ctx.logError(s"Cannot determine fingerprint generation algorithm.")
            throw new InvalidKeyException(s"${archivePath.getPath} was exported with different key fingerprint: $archiveFingerprint. The fingerprint generation algorithm cannot be determined.")
        }

        if (currentFingerprint != archiveFingerprint) {
          throw new InvalidKeyException(s"${archivePath.getPath} was exported with different key fingerprint: $archiveFingerprint. Fingerprint of this server: $currentFingerprint")
        }

        ctx.logOutput(s"Starting repository import from [${archivePath.getPath}]")
        val exportedCiSections = rootElement.getChildren(exportedConfigurationItems, XlNamespace)

        val joined = exportedCiSections.asScala.fold(new Element(exportedRootId))((newElement, sectionElement) => {
          newElement.addContent(sectionElement.removeContent())
        })

        ctx.logOutput(s"Converting [${joined.getContentSize}] XML nodes into CIs")
        val exportedCisReader = new JDom2Reader(joined)

        val converter = new ArtifactExtractingCiConverter(zipArchive, exportedCisReader)

        val cis = converter.readCis(new CiXstreamReader(exportedCisReader))
        ctx.logOutput(s"Finished deserialization of [${cis.size()}] CIs.")
        converter.resolveReferences(ctx.getRepository)
        ctx.logOutput(s"Finished references resolution.")

        exportedCiSections
          .asScala
          .map(_.getAttribute(exportedRootId).getValue)
          .foreach { sectionRoot =>
              repositoryService.deleteBeforeImport(sectionRoot)
              ctx.logOutput(s"Preparing [$sectionRoot] for import")
          }

        ctx.logOutput(s"Storing CIs in the repository.")
        repositoryService.create(cis.asScala.toSeq :_*)
        ctx.logOutput(s"Successfully stored ${cis.size()} in the repository.")
      }
    }
  }

  private class ArtifactExtractingCiConverter(zip: ZipFile, outerReader: JDom2Reader) extends PasswordEncryptingCiConverter {

    override def readCi(reader: CiReader): ConfigurationItem = {
      val workDir = WorkDirContext.get()
      trace(s"Processing [${reader.getId}]")
      super.readCi(reader) match {
        case artifact: SourceArtifact if isStoredArtifact(artifact) =>
          val artifactInArchive = outerReader.getAttribute("file")
          trace(s"Attaching artifact file [$artifactInArchive}] to [${reader.getId}].")
          artifact.setFile(extractedFile(artifactInArchive, zip, workDir))
          artifact
        case artifact: SourceArtifact if isExternalArtifact(artifact) =>
          artifact.setFile(downloadResolvedFile(artifact, workDir))
          artifact
        case ci => ci
      }
    }

  }

  private def extractedFile(entryPath: String, archive: ZipFile, workDir: WorkDir): OverthereFile = {
    val artifactWorkPath = (Paths.get(workDir.getPath) / entryPath).createFile()

    using(archive.entryStream(entryPath)) { entryStream =>
      using(new FileOutputStream(artifactWorkPath)) { aos =>
        OverthereUtils.write(entryStream, aos)
      }
    }

    LocalFile.valueOf(artifactWorkPath)
  }
}
