package com.xebialabs.xlplatform.extensions.exportcis

import java.nio.file.Path

import com.xebialabs.deployit.engine.api.ServiceHolder
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.repository.RepositoryServiceHolder._
import com.xebialabs.deployit.repository.WorkDirContext
import com.xebialabs.xlplatform.endpoints.{AuthenticatedData, ServicesSupport}
import com.xebialabs.xlplatform.extensions.exportcis.archive.RepositoryExporter
import com.xebialabs.xlplatform.sugar.PathSugar._
import grizzled.slf4j.Logging
import org.joda.time.DateTime
import org.joda.time.format.DateTimeFormat

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success}

trait CisExportSupport extends ServicesSupport with Logging {

  def export(rootCiId: String, destinationPath: Path)(implicit auth: AuthenticatedData): String = {
    WorkDirContext.initWorkdir("export")
    try {
      destinationPath.createDirectory(failIfExists = false)
      val date = DateTimeFormat.forPattern("yyyy-MM-dd_HH-mm-ss_sss").print(DateTime.now)
      val exportFileName = s"${rootCiId.split("/").lastOption.getOrElse("Root")}_$date.zip"
      val exportFile = destinationPath / exportFileName
      val version = ServiceHolder.getServerService.getInfo.getVersion

      taskCreationService.createWithStepBlock(auth.toAuthentication,
        s"Export CI tree starting from '$rootCiId' into '${destinationPath.path}'",
        List[Step](ExportStep(rootCiId, exportFile, auth.username, version)).asJava,
        Map("exportedFile" -> exportFile.path).asJava)
    } finally {
      WorkDirContext.clear()
    }
  }

}

case class ExportStep(rootCiId: String, toFile: Path, username: String,
                      version: String) extends Step with Logging {

  override def execute(ctx: ExecutionContext): StepExitCode = {
    val exporter = new RepositoryExporter(getRepositoryService, ctx)
    exporter.export(List(rootCiId), toFile, username, version) match {
      case Success(_) => StepExitCode.SUCCESS
      case Failure(e) =>
        logger.error(s"Failed to export $rootCiId to $toFile", e)
        StepExitCode.FAIL
    }
}

  override def getDescription: String = "Export configuration items into a ZIP file"
  override def getOrder: Int = Step.DEFAULT_ORDER
}
