/*
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS
 * FOR A PARTICULAR PURPOSE. THIS CODE AND INFORMATION ARE NOT SUPPORTED BY XEBIALABS.
 */
package com.xebialabs.gradle.plugins.xlrelease.tasks

import com.google.common.annotations.VisibleForTesting
import com.xebialabs.gradle.plugins.xlrelease.XlReleaseInstancePlugin
import org.gradle.api.DefaultTask
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.TaskAction

import static com.xebialabs.gradle.plugins.xlrelease.tasks.StopXlReleaseTask.waitForUrl

/**
 * Gradle task which starts an XL Release instance with current plugin included.
 */
class StartXlReleaseTask extends DefaultTask {

  /**
   * If true then original resources from src/main/resources will be added to XL Release classpath
   * instead of processed ones from build/. This might be handy to quickly change python scripts
   * without need to rebuild and restart the project.
   */
  @Input
  def useSourcesDirectly

  @Input
  def xlReleaseHome

  @Input
  def xlReleasePort

  @Input
  def xlReleaseLicense

  @TaskAction
  @SuppressWarnings("GroovyUnusedDeclaration")
  public void startXLRelease() {

    StopXlReleaseTask.stopXLRelease(logger, "$xlReleasePort")

    logger.lifecycle("Starting XL Release server ...")

    def configDefaultsFile = project.file("${project.buildDir}/server/conf/xl-release-server.conf.defaults")
    configDefaultsFile.getParentFile().mkdirs()
    configDefaultsFile.createNewFile()
    configDefaultsFile.write(CONFIG_DEFAULTS.concat("\nhttp.port=$xlReleasePort"))

    def classpath = [
        "${project.buildDir}/server/conf",
        "$xlReleaseHome/conf",
        "$xlReleaseHome/ext",
        "$xlReleaseHome/hotfix/*",
        "$xlReleaseHome/lib/*",
        "$xlReleaseHome/plugins/*"
    ]
    classpath += [
        "${project.buildDir}/classes/test",
        "${project.buildDir}/classes/main"
    ].findAll { project.file(it).exists() }

    if (Boolean.valueOf("$useSourcesDirectly")) {
      classpath += [
          "${project.projectDir}/src/test/resources",
          "${project.projectDir}/src/main/resources",
      ].findAll { project.file(it).exists() }
    } else {
      classpath += [
          "${project.buildDir}/resources/test",
          "${project.buildDir}/resources/main"
      ].findAll { project.file(it).exists() }
    }

    project.configurations.findByName(XlReleaseInstancePlugin.ADDITIONAL_CLASSPATH_CONFIGURATION).each { file ->
      classpath.add("$file")
    }

    def workdir = "${project.buildDir}/server/"
    logger.info("Starting XL Release in working directory: $workdir")
    logger.info("Using following classpath to start XL Release: $classpath")

    def antJavaArgs = [
        fork: true,
        spawn: true,
        classpath: classpath.join(':'),
        dir: workdir,
        classname: "com.xebialabs.xlrelease.XLReleaseBootstrapper"
    ]

    def jvmArgs = [
      "-Xmx1024m",
      "-XX:MaxPermSize=256m",
      "-Djava.awt.headless=true",
      "-Dlogback.configurationFile=$xlReleaseHome/conf/logback.xml",
      "-Dderby.stream.error.file=log/derby.log",
      "-Djava.security.manager=java.lang.SecurityManager",
      "-Djava.security.policy=$xlReleaseHome/conf/xl-release.policy",
      "-Djava.net.preferIPv4Stack=true",
      "-Dxl.license.location=${project.file(xlReleaseLicense)}",
      "-XX:-OmitStackTraceInFastThrow",
      "-Xverify:none"
    ]
    if (project.hasProperty("debug")) {
      logger.lifecycle("Enabled debug mode on port 5005")
      jvmArgs += "-Xdebug"
      // TODO: make suspend=y and port configurable
      jvmArgs += "-Xrunjdwp:transport=dt_socket,server=y,suspend=n,address=5005"
    }

    def programArgs = [
      "-reinitialize",
      "-force",
      "-force-upgrades",
      "-setup-defaults",
      "${configDefaultsFile}"
    ]

    startMethod(antJavaArgs, jvmArgs, programArgs)

    // TODO: make port and context root configurable
    def url = "http://localhost:$xlReleasePort"
    waitMethod(url)

    logger.lifecycle("XL Release has started at $url")
  }

  @VisibleForTesting
  protected def startMethod = { Map<String, Object> antJavaArgs, Collection<String> jvmArgs, Collection<String> programArgs ->
    ant.java(antJavaArgs) {
      jvmArgs.each {
        jvmarg(value: it)
      }
      programArgs.each {
        arg(value: it)
      }
    }
  }

  @VisibleForTesting
  protected def waitMethod = { String url ->
    waitForUrl(url, true, logger)
  }

  // TODO: Make this template configurable
  private static final String CONFIG_DEFAULTS = """\
    # XL Release configuration file.
    admin.password={b64}YFKOzMTEICsqFJ592l2FbQ\\=\\=
    jcr.repository.path=repository
    threads.min=3
    ssl=false
    client.session.remember.enabled=true
    http.bind.address=0.0.0.0
    http.context.root=/
    threads.max=24
    client.session.timeout.minutes=0
    http.context.root=/
    hide.internals=false
    importable.packages.path=importablePackages""".stripIndent()

}
