package com.xebialabs.gradle.documentation.restdoc.doclet;

import com.google.common.base.Strings;
import com.sun.source.doctree.DocTree;
import com.sun.source.doctree.SeeTree;

import com.xebialabs.gradle.plugins.restdoclet.doclet.scanner.FirstChildrenScanner;
import com.xebialabs.gradle.plugins.restdoclet.doclet.scanner.MethodScanner;
import com.xebialabs.gradle.plugins.restdoclet.doclet.scanner.StringKindComparator;

import javax.lang.model.element.*;
import javax.lang.model.type.TypeMirror;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import static java.util.stream.Collectors.toList;

/**
 * Writes a documentation page for a single service class.
 */
public class RestServiceWriter extends RestDocWriter {

    private static final List<String> PATH_PARAM = List.of("javax.ws.rs.PathParam", "jakarta.ws.rs.PathParam");
    private static final List<String> QUERY_PARAM = List.of("javax.ws.rs.QueryParam", "jakarta.ws.rs.QueryParam");
    private static final List<String> HEADER_PARAM = List.of("javax.ws.rs.HeaderParam", "jakarta.ws.rs.HeaderParam");
    private static final List<String> GET = List.of("javax.ws.rs.GET", "jakarta.ws.rs.GET");
    private static final List<String> POST = List.of("javax.ws.rs.POST", "jakarta.ws.rs.POST");
    private static final List<String> PUT = List.of("javax.ws.rs.PUT", "jakarta.ws.rs.PUT");
    private static final List<String> DELETE = List.of("javax.ws.rs.DELETE", "jakarta.ws.rs.DELETE");
    private static final List<String> HEAD = List.of("javax.ws.rs.HEAD", "jakarta.ws.rs.HEAD");

    private final TypeElement service;
    private final boolean printJsonContentType;
    private final String path;
    private final String defaultConsumes;
    private final String defaultProduces;

    /**
     * Creates a new writer. Use a new instance for each page.
     *
     * @param writer               the stream to write to.
     * @param service              the javadoc info of the Rest class to be documented.
     * @param printJsonContentType
     */
    public RestServiceWriter(PrintWriter writer, TypeElement service, boolean printJsonContentType) {
        super(writer);
        this.service = service;
        this.printJsonContentType = printJsonContentType;
        this.path = getPath(service);
        this.defaultConsumes = getConsumes(service);
        this.defaultProduces = getProduces(service);
    }

    //
    // Page structure
    //

    /**
     * Writes the entire page.
     */
    public void writeRestService() {
        writeHeader();
        writeIndex();
        writeMethodDetails();
    }

    //
    // Html rendering
    //

    private void writeHeader() {
        div(h1(service.getSimpleName().toString())).cssClass("manual-title").write();
        writeDeprecationDetails(service);
        p(asText(getEntireComment(service)) + parseRestDetails(service)).write();

    }

    private void writeIndex() {
        table().cssClass("parameter-table").writeOpen();
        for (ExecutableElement method : getRestMethods(service)) {
            String httpMethod = getHttpMethod(method);
            row(httpMethod, link("#" + getAnchor(method), getUri(method)), asText(getDocTree(method))).write();
        }
        table().writeClose();
    }

    private void writeMethodDetails() {
        hr().write();
        for (ExecutableElement method : getRestMethods(service)) {
            writeMethodDetail(method);
        }
    }

    private void writeMethodDetail(ExecutableElement method) {

        // Method signature and comments
        anchor(getAnchor(method)).write();
        h2(getHttpMethod(method), " ", getUri(method)).cssClass("resource-header").write();
        p(asText(getDeprecatedTags(method))).write();

        String restDetails = parseRestDetails(method);

        div(asText(getDocTree(method)) + restDetails).write();

        // Permissions
        writePermissions(method);

        //Headers
        writeHeaders(method);

        // Parameters
        writeParameters(method);

        // Return type
        writeReturnType(method);

        List<? extends DocTree> seeDocTree = getSeeTags(method);
        // See tags
        for (DocTree seeTag : seeDocTree) {
            List<SeeTree> seeTreeReferenceList = (List<SeeTree>) ((SeeTree) seeTag).getReference();
            definitionList("See", asText(seeTreeReferenceList)).write();
        }
    }

    private void writeHeaders(ExecutableElement method) {
        List<DocTree> headerTags = getHeadersTags(method);
        com.xebialabs.commons.html.Element dt = definitionList("Headers");
        if (!headerTags.isEmpty()) {
            for (DocTree header : headerTags) {
                String textAfterTag = getTextAfterTag(header, "@headers");
                String firstWord = firstWord(textAfterTag);
                String restOfSentence = textAfterTag.substring(firstWord.length());
                dt.add(element("dd", italic(firstWord), restOfSentence));
            }
            dt.write();
        }
    }

    private void writePermissions(ExecutableElement method) {
        List<? extends DocTree> permissionDocTree = getPermissionTags(method);
        if (!permissionDocTree.isEmpty()) {
            com.xebialabs.commons.html.Element dt = definitionList("Permissions");
            for (DocTree permission : permissionDocTree) {
                String textAfterTag = getTextAfterTag(permission, "@permission");
                String firstWord = firstWord(textAfterTag);
                String rest = textAfterTag.substring(firstWord.length());
                if (!Strings.isNullOrEmpty(rest)) {
                    rest = " - " + rest;
                }
                dt.add(element("dd", code(firstWord), rest));
            }
            dt.write();
        }
    }

    private String getTextAfterTag(DocTree header, String tagKind) {
        FirstChildrenScanner scanner = new FirstChildrenScanner(new StringKindComparator(tagKind));
        scanner.scan(header, false);
        return scanner.getTags().get(0).toString();
    }

    private String parseRestDetails(Element element) {
        List<DocTree> restDetails = getRestDetailsTags(element);
        if (restDetails.isEmpty()) {
            return "";
        } else {
            return " " + asText(restDetails);
        }
    }

    private void writeDeprecationDetails(Element element) {
        com.xebialabs.commons.html.Element p = p(element("strong", "Deprecated:"));
        List<DocTree> deprecatedTags = getDeprecatedTags(element);
        if (!deprecatedTags.isEmpty()) {
            p.add(italic(asText(deprecatedTags)));
            p.write();
        }
    }

    private void writeReturnType(ExecutableElement method) {

        if ("void".equals(method.getReturnType().toString())) {
            definitionList("Response body", italic("Empty")).write();
        } else {
            definitionList(
                    "Response body",
                    renderType(method.getReturnType()) + getReturnTypeInfo(method),
                    "Content type: " + getMethodProduces(method)
            ).write();
        }
    }

    private String getReturnTypeInfo(ExecutableElement method) {
        List<DocTree> returnTypeDocTree = getReturnTags(method);
        if (returnTypeDocTree.isEmpty()) {
            return "";
        } else {
            return " - " + asText(returnTypeDocTree);
        }
    }

    private void writeParameters(ExecutableElement method) {
        List<? extends VariableElement> paramElements = method.getParameters();
        if (paramElements.isEmpty()) {
            return;
        }

        com.xebialabs.commons.html.Element table = table().cssClass("parameter-table");
        for (VariableElement paramElement : paramElements) { // see above
            ParameterInfo info = getParameterInfo(method, paramElement);
            if (info == null) {
                System.out.println("Warning: No actual parameter for @param " + paramElement.getSimpleName().toString() + " on " + method);
                continue;
            }
            List<DocTree> allParamTreeList = getParamTags(method);
            List<DocTree> filteredElementForRow = new ArrayList<>();
            if (!allParamTreeList.isEmpty()) {
                filteredElementForRow = allParamTreeList.stream().filter(p -> p.toString().contains(paramElement.toString())).collect(toList());
            }
            table.add(row(italic(info.kind), info.name, renderType(info.type), asText(filteredElementForRow)));
        }
        definitionList("Parameters", table).write();
    }

    //
    // Inspection
    //

    private static List<ExecutableElement> getRestMethods(TypeElement service) {
        MethodScanner scanner = new MethodScanner();
        scanner.scan(service, null);

        return scanner.getMethods().stream().filter(RestServiceWriter::isRestMethod).sorted(new MethodComparator()).collect(toList());
    }

    //
    private static boolean isRestMethod(Element element) {
        List<? extends AnnotationMirror> mirrors = element.getAnnotationMirrors();
        for (AnnotationMirror mirror : mirrors) {
            String annotationType = mirror.getAnnotationType().toString();
            if (annotationType.startsWith("javax.ws.rs") || annotationType.startsWith("jakarta.ws.rs")) {
                return true;
            }
        }
        return false;
    }

    private static String getPath(Element element) {
        String javaxAnnotation = getAnnotationValue(element, "javax.ws.rs.Path");
        String jakartaAnnotation = getAnnotationValue(element, "jakarta.ws.rs.Path");

        if (javaxAnnotation != null && !javaxAnnotation.isBlank()) {
            return javaxAnnotation;
        } else {
            return jakartaAnnotation;
        }
    }

    private String getUri(Element element) {
        return path + "/" + getPath(element);
    }

    //
    private String getAnchor(Element element) {
        return getUri(element) + ":" + getHttpMethod(element);
    }

    //
    private String getConsumes(Element element) {
        String javaxValue = getAnnotationValue(element, "javax.ws.rs.Consumes");
        String jakartaValue = getAnnotationValue(element, "jakarta.ws.rs.Consumes");

        String annValue = javaxValue != null && !javaxValue.isBlank() ? javaxValue : jakartaValue;

        return optionallyStripJson(annValue.replace("\"", "").replaceAll("[\"\\]\\[]", ""));
    }

    private String optionallyStripJson(String s) {
        if (!printJsonContentType) {
            return s.replace("application/json", "").replaceAll(",\\s*$", "");
        }
        return s;
    }

    private String getMethodConsumes(ExecutableElement method) {
        String consumes = getConsumes(method);
        if (Strings.isNullOrEmpty(consumes)) {
            return defaultConsumes;
        }
        return consumes;
    }

    private String getProduces(Element element) {
        String javaxValue = getAnnotationValue(element, "javax.ws.rs.Produces");
        String jakartaValue = getAnnotationValue(element, "jakarta.ws.rs.Produces");

        String annValue = javaxValue != null && !javaxValue.isBlank() ? javaxValue : jakartaValue;

        return optionallyStripJson(annValue.replaceAll("[\"\\]\\[]", ""));
    }

    private String getMethodProduces(ExecutableElement method) {
        String produces = getProduces(method);
        if (Strings.isNullOrEmpty(produces)) {
            return defaultProduces;
        }
        return produces;
    }

    private static String getHttpMethod(Element element) {
        for (AnnotationMirror annotation : element.getAnnotationMirrors()) {
            String annotationType = annotation.getAnnotationType().toString();
            if (GET.contains(annotationType)) {
                return "GET";
            }
            if (POST.contains(annotationType)) {
                return "POST";
            }
            if (PUT.contains(annotationType)) {
                return "PUT";
            }
            if (DELETE.contains(annotationType)) {
                return "DELETE";
            }
            if (HEAD.contains(annotationType)) {
                return "HEAD";
            }
        }
        return "?";
    }

    //
    public static String getAnnotationValue(Element element, String annotationType) {
        return getAnnotationValue(getAnnotation(element, annotationType));
    }

    private static String getAnnotationValue(AnnotationMirror annotation) {
        if (annotation == null) {
            return "";
        }

        for (AnnotationValue item : annotation.getElementValues().values()) {
            Object value = item.getValue();
            if (value instanceof Object[]) {
                return Arrays.asList((Object[]) value).toString();
            }
            return value.toString();
        }

        return "";
    }

    private static AnnotationMirror getAnnotation(Element element, String type) {
        return element.getAnnotationMirrors().stream().filter(mirror -> mirror.getAnnotationType().toString().equals(type)).findFirst().orElse(null);
    }

    private ParameterInfo getParameterInfo(ExecutableElement method, VariableElement tag) {

        List<? extends VariableElement> parameters = method.getParameters();
        String name = tag.getSimpleName().toString();
        for (VariableElement param : parameters) {
            if (!param.getSimpleName().toString().equals(name)) {
                continue;
            }

            TypeMirror type = param.asType();
            for (AnnotationMirror annotation : param.getAnnotationMirrors()) {
                String annotationType = annotation.getAnnotationType().toString();
                if (PATH_PARAM.contains(annotationType)) {
                    return new ParameterInfo(getAnnotationValue(annotation), "Path", type);
                }
                if (QUERY_PARAM.contains(annotationType)) {
                    return new ParameterInfo(getAnnotationValue(annotation), "Query", type);
                }
                if (HEADER_PARAM.contains(annotationType)) {
                    return new ParameterInfo(getAnnotationValue(annotation), "Header", type);
                }
                if (annotationType.equals("org.jboss.resteasy.annotations.providers.multipart.MultipartForm")) {
                    return new ParameterInfo(getAnnotationValue(annotation), "Multipart", type);
                }
            }
            return new ParameterInfo(getMethodConsumes(method), "Request&nbsp;body", type);
        }

        return null;
    }

    private static class ParameterInfo {
        final String name;
        final String kind;
        final TypeMirror type;

        ParameterInfo(String name, String kind, TypeMirror type) {
            this.name = name;
            this.kind = kind;
            this.type = type;
        }
    }

    private static class MethodComparator implements Comparator<ExecutableElement> {

        @Override
        public int compare(ExecutableElement method, ExecutableElement anotherMethod) {
            return getPath(method).compareTo(getPath(anotherMethod));
        }
    }
}
