package com.xebialabs.gradle.documentation.restdoc.doclet;

import com.google.common.base.Preconditions;
import com.xebialabs.gradle.plugins.restdoclet.doclet.DocEnv;
import com.xebialabs.gradle.plugins.restdoclet.doclet.DocletUtils;
import jdk.javadoc.doclet.Doclet;
import jdk.javadoc.doclet.DocletEnvironment;
import jdk.javadoc.doclet.StandardDoclet;

import javax.lang.model.element.TypeElement;
import java.io.*;
import java.util.*;

import static com.google.common.collect.Lists.newArrayList;
import static java.util.stream.Collectors.toSet;

public class RestDoclet extends StandardDoclet {

    private File destinationDir = new File(".");
    private boolean showJsonSupport = true;
    private String productName = "XL-Release";
    private String jsonOutputFile = "api.json";

    @Override
    public Set<Doclet.Option> getSupportedOptions() {
        Set<? extends Option> defaults = super.getSupportedOptions();
        Set<Option> options = getOptions();

        Set<String> optionNames = options.stream().map(opt -> opt.getNames().get(0)).collect(toSet());
        Set<Option> filtered = defaults.stream().filter(opt -> Collections.disjoint(opt.getNames(), optionNames)).collect(toSet());

        options.addAll(filtered);
        return options;
    }

    private Set<Option> getOptions() {
        Set<Option> options = new HashSet<>();
        options.add(new StandardOption("-showJsonSupport", this::setShowJsonSupport));
        options.add(new StandardOption("-product", this::setProductName));
        options.add(new StandardOption("-jsonOutputFile", this::setJsonOutputFile));
        options.add(new StandardOption("-d", this::setDestinationDir));
        return options;
    }

    @Override
    public boolean run(DocletEnvironment docEnv) {
        DocEnv.set(docEnv);
        List<TypeElement> apiServices = DocletUtils.findAnnotatedClasses(docEnv, "com.xebialabs.xlplatform.documentation.PublicApi");

        List<TypeElement> apiRefs = DocletUtils.findAnnotatedClasses(docEnv, "com.xebialabs.xlplatform.documentation.PublicApiRef");

        List<TypeElement> allApiClasses = newArrayList(apiServices);
        allApiClasses.addAll(apiRefs);

        Preconditions.checkNotNull(productName, "The productName should not be 'null', please specify '-product'");

        if (!apiServices.isEmpty()) {
            copyCss();
            writeOverview(apiServices);
            writeServices(apiServices);
            writeLinkedFiles();
        }
        writeJsonFile(allApiClasses);

        return true;
    }

    private void copyCss() {
        Resource.fromClasspath("restdoc/layout.css").copy(new File(destinationDir, "layout.css"));
        Resource.fromClasspath("restdoc/restdoc.css").copy(new File(destinationDir, "restdoc.css"));
        Resource.fromClasspath("restdoc/image.zip").unzip(destinationDir);
    }

    private void writeJsonFile(List<TypeElement> restServices) {
        File destination = new File(destinationDir, jsonOutputFile);
        try (FileWriter fileWriter = new FileWriter(destination)) {
            restServices.sort(Comparator.comparing(l -> l.getSimpleName().toString()));
            fileWriter.write(DocletUtils.generateJson(restServices));
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    private void writeOverview(List<TypeElement> restServices) {
        PrintWriter writer = openFile("index.html", String.format("%s REST API", productName));
        try {
            new OverviewWriter(writer, productName).write(restServices);
        } finally {
            closeFile(writer);
        }
    }

    private void writeServices(List<TypeElement> restServices) {
        // Register files that are going to be generated, for cross-referencing
        for (TypeElement service : restServices) {
            FileCatalog.SINGLETON.add(fileNameFor(service.getSimpleName().toString()));
        }

        // Generate documentation per service
        for (TypeElement service : restServices) {
            String simpleName = service.getSimpleName().toString();
            PrintWriter writer = openFile(fileNameFor(simpleName), simpleName);
            try {
                new RestServiceWriter(writer, service, showJsonSupport).writeRestService();
            } finally {
                closeFile(writer);
            }
        }
    }


    private void writeLinkedFiles() {
        for (String item : FileCatalog.SINGLETON.getItems()) {
            if (FileCatalog.isResource(item)) {
                PrintWriter writer = openFile(item, item.replace(".html", ""));
                try {
                    FileCatalog.write(item, writer);
                } finally {
                    closeFile(writer);
                }
            }
        }

        for (String item : FileCatalog.SINGLETON.getMissing()) {
            System.out.println("Missing cross reference: " + item);
        }
    }

    public static String fileNameFor(String service) {
        if (service == null) {
            return null;
        }
        return service + ".html";
    }

    public PrintWriter openFile(String fileName, String title) {
        File file = new File(destinationDir, fileName);
        try {
            PrintWriter writer = new PrintWriter(file);
            new PageTemplate(writer).writeHeader(title);
            return writer;
        } catch (FileNotFoundException e) {
            throw new IllegalStateException("Can't open " + file.getAbsolutePath() + " for writing.");
        }
    }

    public static void closeFile(PrintWriter writer) {
        new PageTemplate(writer).writeFooter();
        writer.flush();
        writer.close();
    }

    public void setDestinationDir(String destinationDir) {
        this.destinationDir = new File(destinationDir);
    }

    public void setShowJsonSupport(String showJsonSupport) {
        this.showJsonSupport = Boolean.parseBoolean(showJsonSupport);
    }

    public void setProductName(String productName) {
        this.productName = productName;
    }

    public void setJsonOutputFile(String jsonOutputFile) {
        this.jsonOutputFile = jsonOutputFile;
    }
}
