package com.xebialabs.gradle.documentation.restdoc.doclet;

import com.google.common.base.Strings;
import com.sun.source.doctree.*;
import com.xebialabs.commons.html.HtmlWriter;
import com.xebialabs.gradle.plugins.restdoclet.doclet.DocEnv;

import javax.lang.model.element.Element;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeMirror;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import static java.util.stream.Collectors.toList;

/**
 * HtmlWriter with convenience methods for writing REST documentation.
 */
public class RestDocWriter extends HtmlWriter {

    private final FileCatalog fileCatalog;

    public RestDocWriter(PrintWriter writer) {
        super(writer);
        fileCatalog = FileCatalog.SINGLETON;
    }

    protected String asText(List<? extends DocTree> docTreeList) {
        StringBuilder builder = new StringBuilder();

        if (!docTreeList.isEmpty()) {
            for (DocTree docTree : docTreeList) {
                if (docTree instanceof SeeTree) {
                    appendLink(builder, docTree);
                } else if (docTree instanceof ReferenceTree) {
                    appendReferenceText(builder, docTree);
                } else if (docTree instanceof ParamTree) {
                    ((ParamTree) docTree).getDescription().stream().filter(Objects::nonNull).forEach(p -> checkAndAppendLink(p, builder));
                } else if (docTree instanceof LiteralTree) {
                    builder.append(code(((LiteralTree) docTree).getBody()));
                } else if (docTree instanceof DeprecatedTree) {
                    builder.append(span("(Deprecated)").cssClass("deprecated"));
                } else if (docTree instanceof UnknownBlockTagTree) {
                   ((UnknownBlockTagTree) docTree).getContent().stream().filter(Objects::nonNull).forEach(p -> checkAndAppendLink(p, builder));
                } else if (docTree instanceof ReturnTree) {
                    ((ReturnTree) docTree).getDescription().stream().filter(Objects::nonNull).forEach(p -> checkAndAppendLink(p, builder));
                } else if (docTree instanceof LinkTree) {
                    builder.append(((LinkTree) docTree).getReference());
                } else if (docTree instanceof TextTree) {
                    builder.append(((TextTree) docTree).getBody());
                } else {
                    if (builder.length() > 0) {
                        builder.append(((TextTree) docTree).getBody());
                    } else {
                        builder.append("");
                    }
                }
            }
        }
        return builder.toString();
    }


    protected void checkAndAppendLink(DocTree docTree, StringBuilder builder){
        if(docTree.toString().contains("@link")){
            appendLink(builder, docTree);
        } else {
            builder.append(docTree);
        }
    }

    protected String asText(TypeMirror typeMirror) {
        StringBuilder builder = new StringBuilder();
        builder.append(typeMirror.toString());

        String separator = " of ";
        for (TypeMirror paramType : getParametrizedTypes(typeMirror)) {
            builder.append(separator);
            builder.append(paramType.toString());
            separator = ", ";
        }

        return builder.toString();
    }

    private void appendReferenceText(StringBuilder builder, DocTree docTree) {
        ReferenceTree linkTree = (ReferenceTree) docTree;
        String text = linkTree.getSignature();
        builder.append(text);
    }

    private void appendLink(StringBuilder builder, DocTree docTree) {
        LinkTree linkTree = (LinkTree) docTree;
        String file = RestDoclet.fileNameFor(linkTree.getReference().getSignature());
        String text = linkTree.getReference().getSignature();

        if (Strings.isNullOrEmpty(text)) {
            text = "";
        }

        if (fileCatalog.check(file)) {
            builder.append(link(file, text));
        } else {
            builder.append(bold(text));
        }
    }

    public static String firstWord(String sentence) {
        return sentence.split("\\s")[0];
    }

    protected String asReference(TypeMirror typeMirror) {
        List<TypeMirror> typeMirrorList = getParametrizedTypes(typeMirror);

        // Default case: fully qualified name of non-parametrized type.
        if (typeMirrorList.isEmpty()) {
            return typeMirror.getKind() + ".html";
        }

        // Cook something up for 'List of'
        return typeMirrorList.get(0) + "-" + typeMirror.getKind().name() + ".html";
    }

    public List<TypeMirror> getParametrizedTypes(TypeMirror typeMirror) {

        List<TypeMirror> types = new ArrayList<>();

        if (typeMirror instanceof DeclaredType) {
            types.addAll(((DeclaredType) typeMirror).getTypeArguments());
        }

        return types;
    }

    protected Object renderType(TypeMirror typeMirror) {
        Object returnTypeText = asText(typeMirror);
        String externalFile = asReference(typeMirror);

        // Add references to catalog
        if (fileCatalog.check(externalFile)) {
            returnTypeText = link(externalFile, returnTypeText);
        }
        for (TypeMirror paramType : getParametrizedTypes(typeMirror)) {
            fileCatalog.check(asReference(paramType));
        }

        return returnTypeText.toString().substring(returnTypeText.toString().lastIndexOf('.') + 1);
    }

    protected List<DocTree> getDeprecatedTags(Element service) {
        List<DocTree> docTreeList = RestDocletUtils.getBlockTagsDocTree(service);
        if (docTreeList.isEmpty()) {
            return docTreeList;
        } else {
            List<DocTree> deprecatedTreeList = new ArrayList<>(docTreeList);
            return deprecatedTreeList.stream().filter(ele -> ele.getKind().equals(DocTree.Kind.DEPRECATED)).collect(toList());
        }
    }

    protected List<DocTree> getRestDetailsTags(Element service) {
        List<DocTree> docTreeList = RestDocletUtils.getBlockTagsDocTree(service);
        if (docTreeList.isEmpty()) {
            return docTreeList;
        } else {
            List<DocTree> restDetailsDocTreeList = new ArrayList<>(docTreeList);
            return restDetailsDocTreeList.stream().filter(ele -> ele.toString().contains("@restDetails")).collect(toList());
        }
    }

    protected List<? extends DocTree> getPermissionTags(Element service) {
        DocCommentTree commentsTree = DocEnv.get().getDocTrees().getDocCommentTree(service);
        List<? extends DocTree> blockTags = commentsTree != null ? commentsTree.getBlockTags() : new ArrayList<>();

        if (blockTags.isEmpty()) {
            return blockTags;
        } else {
            return blockTags.stream().filter(tag -> tag.toString().contains("@permission")).collect(toList());
        }
    }

    protected List<DocTree> getHeadersTags(Element service) {
        List<DocTree> docTreeList = RestDocletUtils.getBlockTagsDocTree(service);
        if (docTreeList.isEmpty()) {
            return docTreeList;
        } else {
            List<DocTree> restDetailsDocTreeList = new ArrayList<>(docTreeList);
            return restDetailsDocTreeList.stream().filter(ele -> ele.toString().contains("@headers")).collect(toList());
        }
    }

    protected List<DocTree> getReturnTags(Element service) {
        List<DocTree> docTreeList = RestDocletUtils.getBlockTagsDocTree(service);
        if (docTreeList.isEmpty()) {
            return docTreeList;
        } else {
            List<DocTree> deprecatedTreeList = new ArrayList<>(docTreeList);
            return deprecatedTreeList.stream().filter(ele -> ele.getKind().equals(DocTree.Kind.RETURN)).collect(toList());
        }
    }

    protected List<DocTree> getSeeTags(Element service) {
        List<DocTree> docTreeList = RestDocletUtils.getBlockTagsDocTree(service);
        if (docTreeList.isEmpty()) {
            return docTreeList;
        } else {
            List<DocTree> deprecatedTreeList = new ArrayList<>(docTreeList);
            return deprecatedTreeList.stream().filter(ele -> ele.getKind().equals(DocTree.Kind.SEE)).collect(toList());
        }
    }

    protected List<DocTree> getParamTags(Element service) {
        List<DocTree> docTreeList = RestDocletUtils.getBlockTagsDocTree(service);
        if (docTreeList.isEmpty()) {
            return docTreeList;
        } else {
            List<DocTree> deprecatedTreeList = new ArrayList<>(docTreeList);
            return deprecatedTreeList.stream().filter(ele -> ele.getKind().equals(DocTree.Kind.PARAM)).collect(toList());
        }
    }

    protected List<DocTree> getDocTree(Element element) {
        DocCommentTree docCommentTree = DocEnv.get().getDocTrees().getDocCommentTree(element);

        if (docCommentTree == null || docCommentTree.getFirstSentence() == null) {
            return Collections.emptyList();
        }
        List<DocTree> firstSentenceTree = (List<DocTree>) docCommentTree.getFirstSentence();

        if (docCommentTree.getFullBody() != null) {
            firstSentenceTree.addAll(docCommentTree.getFullBody().stream().filter(dt -> DocTree.Kind.DEPRECATED.equals(dt.getKind())).collect(toList()));
        }

        return firstSentenceTree;
    }

    protected List<DocTree> getEntireComment(Element element) {

        DocCommentTree docCommentTree = DocEnv.get().getDocTrees().getDocCommentTree(element);
        if (docCommentTree == null || docCommentTree.getFullBody() == null) {
            return Collections.emptyList();
        }
        return (List<DocTree>) docCommentTree.getFullBody();
    }

}
