package com.xebialabs.gradle.documentation.restdoc.doclet;

import com.google.common.io.ByteSource;
import com.google.common.io.Files;

import java.io.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

/**
 * Utility class for working with files and class path resources.
 */
public abstract class Resource extends ByteSource {
    static final int BUFFER = 2048;

    public void write(PrintWriter writer) {
        try (BufferedReader in = new BufferedReader(new InputStreamReader(openStream()))) {
            String line;
            while ((line = in.readLine()) != null) {
                writer.println(line);
            }
        } catch (IOException e) {
            throw new IllegalArgumentException("Could not read resource " + getName(), e);
        }
    }

    public void unzip(File destinationDir) {
        try {
            destinationDir.mkdirs();
            try (ZipInputStream zis = new ZipInputStream(openBufferedStream())) {
                ZipEntry entry;
                while ((entry = zis.getNextEntry()) != null) {
                    unzipEntry(destinationDir, zis, entry);
                }
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void unzipEntry(File destinationDir, ZipInputStream zis, ZipEntry entry) throws IOException {
        byte data[] = new byte[BUFFER];
        // write the files to the disk
        File file = new File(destinationDir, entry.getName());
        if (entry.isDirectory()) {
            file.mkdirs();
        } else {
            try (OutputStream dest = new BufferedOutputStream(new FileOutputStream(file), BUFFER)) {
                int count;
                while ((count = zis.read(data, 0, BUFFER)) != -1) {
                    dest.write(data, 0, count);
                }
                dest.flush();
            }
        }
    }

    public void copy(File destination) {
        try {
            this.copyTo(Files.asByteSink(destination));
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        }
    }

    public abstract InputStream openStream();

    public abstract String getName();

    public static Resource fromClasspath(final String name) {
        return new Resource() {

            @Override
            public InputStream openStream() {
                return Resource.class.getClassLoader().getResourceAsStream(name);
            }

            @Override
            public String getName() {
                return name;
            }
        };
    }

    public static Resource file(final File file) {
        return new Resource() {

            @Override
            public InputStream openStream() {
                try {
                    return new FileInputStream(file);
                } catch (FileNotFoundException e) {
                    throw new IllegalArgumentException("Can't open " + getName(), e);
                }
            }

            @Override
            public String getName() {
                return file.getAbsolutePath();
            }
        };
    }
}
