package com.xebialabs.gradle.documentation.restdoc.tasks

import groovy.json.JsonSlurper
import org.gradle.api.DefaultTask
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.TaskAction

class CheckJsonJavaDocTask extends DefaultTask {

  @Input
  def inputFiles = [] as Set

  @TaskAction
  def checkMissingLinks() {
    def availableJavaDocs = [] as Set
    def foundReferrals = [] as Set

    for (String jsonFileName : inputFiles) {
      List jsonResult = (List) getJson(project.file(jsonFileName).text)
      availableJavaDocs.addAll(findAvailableJavaDocs(jsonResult))
      foundReferrals.addAll(findReferrals(jsonResult))
    }

    def missingJavaDocs = foundReferrals
    missingJavaDocs.removeAll(availableJavaDocs);

    missingJavaDocs = filterAllClassNamesFromOverthere(missingJavaDocs)

    if (missingJavaDocs) {
      throw new RuntimeException("Missing Java Docs: " + missingJavaDocs.toString() + ". You can fix this issue by marking these classes with @PublicApiRef.")
    }
  }

  static def getJson(String jsonText) {
    new JsonSlurper().parseText(jsonText.toString());
  }

  def findReferrals(List jsonResult) {
    def foundReferrals = []

    //class level
    foundReferrals.addAll(findForClassField("description", jsonResult))
    foundReferrals.addAll(findForClassField("deprecated", jsonResult))

    //method level
    foundReferrals.addAll(findForMethodsField("returnType", jsonResult))
    foundReferrals.addAll(findForMethodsField("description", jsonResult))
    foundReferrals.addAll(findForMethodsField("deprecated", jsonResult))
    foundReferrals.addAll(findForMethodsField("returnDescription", jsonResult))

    //parameter level
    foundReferrals.addAll(findForMethodParametersField("type", jsonResult))
    foundReferrals.addAll(findForMethodParametersField("comment", jsonResult))

    foundReferrals;
  }

  static def iterateBy(List jsonResult, String tagName) {
    return jsonResult.collect { it.get(tagName) }
  }

  static def findAvailableJavaDocs(List jsonResult) {
    iterateBy(jsonResult, "name");
  }

  static def containsXebiaLabsClass(String className) {
    className != null && className.contains("com.xebialabs")
  }

  static def stripTail(String text, String[] oneOf) {
    def symbol = oneOf.find { text.indexOf(it) != -1 }
    if (symbol == null) {
      text
    } else {
      stripTail(text.substring(0, text.indexOf(symbol)), oneOf)
    }
  }

  static def processClassName(String className) {
    String processedClassName = className.substring(className.indexOf("com.xebialabs"))
    processedClassName = stripTail(processedClassName, "#", "}", " ")
    processedClassName.replaceAll(">", "").replaceAll("\\[\\]", "")
  }

  def collectClassNamesFromText(def classNames, String text) {
    if (containsXebiaLabsClass(text)) {
      int startIndex = text.indexOf("com.xebialabs")
      int endIndex = text.indexOf("}", startIndex)

      if (!isHref(text, startIndex)) {
        if (endIndex == -1) {
          classNames.add(processClassName(text.substring(startIndex)))
        } else {
          classNames.add(processClassName(text.substring(startIndex, endIndex)))

          String restOfDescription = text.substring(endIndex)
          if (containsXebiaLabsClass(restOfDescription)) {
            collectClassNamesFromText(classNames, restOfDescription)
          }
        }
      }
    }
    classNames
  }

  def static isHref(def text, def startIndex) {
    return text.substring(0, startIndex).endsWith("href=\"")
  }

  def getClassNamesFromText(String text) {
    collectClassNamesFromText([] as Set, text)
  }

  def findForClassField(String field, List jsonResult) {
    iterateBy(jsonResult, field).
      findAll { containsXebiaLabsClass(it) }.
      collectNested { getClassNamesFromText(it) }.
      flatten()
  }

  def findForMethodsField(String fieldName, List jsonResult) {
    iterateBy(jsonResult, "methods").
      collectNested { item ->
        findAll { containsXebiaLabsClass(item.get(fieldName)) }.
          collectNested { getClassNamesFromText(item.get(fieldName)) }
      }.flatten()
  }

  def findForMethodParametersField(String field, List jsonResult) {
    iterateBy(jsonResult, "methods").
      collectNested { item ->
        item.get("parameters").
          findAll { containsXebiaLabsClass(it.get(field)) }.
          collectNested { getClassNamesFromText(it.get(field)) }
      }.flatten() as Set
  }

  static def filterAllClassNamesFromOverthere(def classNames) {
    classNames.findAll {
      !it.startsWith('com.xebialabs.overthere')
    }
  }

}
