package com.xebialabs.gradle.plugins.restdoclet.doclet

import com.sun.source.doctree.{DocTree, ParamTree}
import com.xebialabs.gradle.plugins.restdoclet.doclet.scanner.{FirstChildrenScanner, MethodArgumentScanner, VariableElementScanner}
import spray.json._

import javax.lang.model.`type`.{NullType, TypeKind, TypeMirror, TypeVariable}
import javax.lang.model.element._
import scala.jdk.CollectionConverters._

trait DocFormats extends CollectionFormats with StandardFormats with AdditionalFormats {

  implicit val methodDocFormat: MethodDocFormat = new MethodDocFormat
  implicit val typeElementFormat: TypeElementFormat = new TypeElementFormat
  implicit val parameterFormat: ParameterFormat = new ParameterFormat
  implicit val fieldDocFormat: FieldDocFormat = new FieldDocFormat
  implicit val constructorDocFormat: ConstructorDocFormat = new ConstructorDocFormat

  val EMPTY_STRING = ""

  private def firstTagText(element: Element, dt: List[DocTree]): JsValue = {
    if (dt.nonEmpty) {
      JsString(DocTreeUtils.textFromTags(element, dt))
    } else {
      JsNull
    }
  }

  private def getDocTree(element: Element) = {
    Option(DocEnv.get().getDocTrees.getDocCommentTree(element)) match {
      case Some(docTree) => docTree.getFullBody.asScala.toList
      case None => List()
    }
  }

  private def deprecationText(element: Element): JsValue = {
    val tags = scanForTagsOfType(DocTree.Kind.DEPRECATED, element)

    firstTagText(element, tags) match {
      case JsNull => getAnnotationsOn(element).find(_.value == "java.lang.Deprecated").map(_ => JsString("")).getOrElse(JsNull)
      case nonNull => nonNull
    }
  }

  private def apiDetailsText(element: Element): String = { // TODO
    val getDocTreeList = getDocTree(element)
    if (getDocTreeList.nonEmpty) {
      DocEnv.get().getDocTrees.getDocCommentTree(element).getBlockTags.asScala.headOption.map(tag => if (tag.toString.contains("@apiDetails")) tag.toString.replaceAll("@apiDetails", EMPTY_STRING)) match {
        case Some(s: String) if s.nonEmpty => return " " + s
        case _ => return ""
      }
    }
    EMPTY_STRING
  }

  private def paramsWithTags(method: ExecutableElement): Seq[ParamWithTag] = {
    val scanner = new MethodArgumentScanner()
    scanner.scan(method)

    val params = scanner.paramDocs.toSeq.map(Some(_)) ++ List.fill(scanner.params.length - scanner.paramDocs.length)(None)
    scanner.params.toList.zip(params)
  }

  def scanForTagsOfType(kind: DocTree.Kind, method: Element): List[DocTree] = {
    scanForTagsOfType(Left(kind), method)
  }

  def scanForTagsOfType(kind: String, method: Element): List[DocTree] = {
    scanForTagsOfType(Right(kind), method)
  }

  def scanForTagsOfType(kind: Either[DocTree.Kind, String], method: Element): List[DocTree] = {
    Option(DocEnv.get().getDocTrees.getDocCommentTree(method)) match {
      case Some(docs) =>
        val scanner = new FirstChildrenScanner(kind)
        scanner.scan(docs, atChildLevel = false)
        scanner.tags.toList
      case None => List()
    }
  }

  def getTypeAsString(typ: TypeMirror): String = {
    typ match {
      case t: TypeVariable if TypeKind.TYPEVAR.equals(t.getKind)
        && t.getLowerBound.isInstanceOf[NullType]
        && t.getUpperBound.getKind.equals(TypeKind.DECLARED)
        && !t.getUpperBound.toString.equals("java.lang.Object") => s"${t.toString} extends ${t.getUpperBound.toString}"
      case _ => typ.toString
    }
  }

  class ConstructorDocFormat extends JsonFormat[ExecutableElement] {
    override def write(constructor: ExecutableElement): JsValue = {
      JsObject(Map(
        "description" -> JsString(DocTreeUtils.textFromTags(constructor, getDocTree(constructor)) + apiDetailsText(constructor)),
        "parameters" -> paramsWithTags(constructor).toJson,
        "deprecated" -> deprecationText(constructor),
        "annotations" -> JsArray(getAnnotationsOn(constructor)),
      ))
    }

    override def read(json: JsValue): ExecutableElement = ???
  }

  class MethodDocFormat extends JsonFormat[ExecutableElement] {

    override def write(element: ExecutableElement): JsValue = {

      val methodGeneric = element.getTypeParameters.asScala.toSeq match {
        case Nil => JsNull
        case params => JsString(s"<${params.map(_.getSimpleName()).mkString(", ")}>")
      }

      val methodDocsOption = Option(DocEnv.get().getDocTrees.getDocCommentTree(element))
      val firstSentence = methodDocsOption match {
        case Some(docs) => docs.getFullBody.asScala.toList
        case None => List()
      }

      JsObject(Map(
        "name" -> JsString(element.getSimpleName.toString),
        "parameters" -> paramsWithTags(element).toJson,
        "methodGeneric" -> methodGeneric,
        "returnType" -> JsString(getTypeAsString(element.getReturnType)),
        "description" -> JsString(DocTreeUtils.textFromTags(element, firstSentence) + apiDetailsText(element)),
        "returnDescription" -> firstTagText(element, scanForTagsOfType(DocTree.Kind.RETURN, element)),
        "deprecated" -> deprecationText(element),
        "permission" -> firstTagText(element, scanForTagsOfType("@permission", element)),
        "header" -> firstTagText(element, scanForTagsOfType("@headers", element)),
        "annotations" -> JsArray(getAnnotationsOn(element)))
      )
    }

    override def read(json: JsValue): ExecutableElement = ???

  }

  def getAnnotationsOn(element: Element): Vector[JsString] = {
    element.getAnnotationMirrors.asScala.map(a => JsString(a.getAnnotationType.toString)).toVector
  }

  class TypeElementFormat extends JsonFormat[TypeElement] {

    def getConstructors(`class`: TypeElement): List[ExecutableElement] = {
      `class`.getKind match {
        case ElementKind.CLASS => `class`.getEnclosedElements.asScala.filter(_.getKind.equals(ElementKind.CONSTRUCTOR))
          .toList.asInstanceOf[List[ExecutableElement]]
        case ElementKind.ENUM | ElementKind.INTERFACE => List()
      }
    }

    def getFields(classElement: TypeElement): List[VariableElement] = {
      val scanner = new VariableElementScanner(ElementKind.FIELD)
      scanner.scan(classElement, 0)
      scanner.elements.toList
    }

    def getConstants(enumElement: TypeElement): Vector[VariableElement] = {
      val scanner = new VariableElementScanner(ElementKind.ENUM_CONSTANT)
      scanner.scan(enumElement, 0)
      scanner.elements.toVector
    }

    override def write(`class`: TypeElement): JsValue = {
      JsObject(Map(
        "name" -> JsString(`class`.getQualifiedName.toString),
        "description" -> JsString(DocTreeUtils.textFromTags(`class`, getDocTree(`class`)) + apiDetailsText(`class`)),
        "methods" -> JythonDocUtils.exposedMethods(`class`).toJson(immSeqFormat(methodDocFormat)),
        "constructors" -> getConstructors(`class`).toJson(listFormat(constructorDocFormat)),
        "fields" -> getFields(`class`).toJson,
        "deprecated" -> deprecationText(`class`),
        "annotations" -> JsArray(getAnnotationsOn(`class`)),
        "constants" -> JsArray(getConstants(`class`).map(c => JsString(c.getSimpleName.toString))),
        "superClass" -> getSuperClassName(`class`)
      ))
    }

    override def read(json: JsValue): TypeElement = ???

    private def getSuperClassName(`class`: TypeElement): JsValue = {
      `class`.getSuperclass.getKind match {
        case TypeKind.NONE => JsNull
        case TypeKind.DECLARED => JsString(DocEnv.get().getTypeUtils.asElement(`class`.getSuperclass).toString)
      }
    }
  }

  class FieldDocFormat extends JsonFormat[VariableElement] {

    override def write(field: VariableElement): JsValue = {
      JsObject(Map(
        "name" -> JsString(field.getSimpleName.toString),
        "type" -> JsString(field.asType().toString),
        "description" -> JsString(DocTreeUtils.textFromTags(field, getDocTree(field))),
        "annotations" -> JsArray(getAnnotationsOn(field))
      ))
    }

    override def read(json: JsValue): VariableElement = ???
  }

  type ParamWithTag = (VariableElement, Option[ParamTree])

  class ParameterFormat extends JsonFormat[ParamWithTag] {

    override def write(obj: ParamWithTag): JsValue = {
      JsObject(Map(
        "name" -> JsString(obj._1.getSimpleName.toString),
        "type" -> JsString(getTypeAsString(obj._1.asType())),
        "comment" -> obj._2.map(pt => JsString(DocTreeUtils.textFromTags(obj._1, pt.getDescription.asScala.toList))).getOrElse(JsNull)
      ))
    }

    override def read(json: JsValue): ParamWithTag = ???
  }

}
