package com.xebialabs.gradle.documentation.restdoc.doclet;

import com.google.common.base.Strings;
import com.sun.source.doctree.DocTree;
import com.sun.source.doctree.SeeTree;

import com.xebialabs.gradle.plugins.restdoclet.doclet.scanner.MethodScanner;

import javax.lang.model.element.*;
import javax.lang.model.type.TypeMirror;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import static java.util.stream.Collectors.toList;

/**
 * Writes a documentation page for a single service class.
 */
public class RestServiceWriter extends RestDocWriter {

    private final TypeElement service;
    private final boolean printJsonContentType;
    private final String path;
    private final String defaultConsumes;
    private final String defaultProduces;

    /**
     * Creates a new writer. Use a new instance for each page.
     *
     * @param writer               the stream to write to.
     * @param service              the javadoc info of the Rest class to be documented.
     * @param printJsonContentType
     */
    public RestServiceWriter(PrintWriter writer, TypeElement service, boolean printJsonContentType) {
        super(writer);
        this.service = service;
        this.printJsonContentType = printJsonContentType;
        this.path = getPath(service);
        this.defaultConsumes = getConsumes(service);
        this.defaultProduces = getProduces(service);
    }

    //
    // Page structure
    //

    /**
     * Writes the entire page.
     */
    public void writeRestService() {
        writeHeader();
        writeIndex();
        writeMethodDetails();
    }

    //
    // Html rendering
    //

    private void writeHeader() {
        div(h1(service.getSimpleName().toString())).cssClass("manual-title").write();
        writeDeprecationDetails(service);
        p(asText(getDocTree(service)) + parseRestDetails(service)).write();

    }

    private void writeIndex() {
        table().cssClass("parameter-table").writeOpen();
        for (ExecutableElement method : getRestMethods(service)) {
            String httpMethod = getHttpMethod(method);
            row(httpMethod, link("#" + getAnchor(method), getUri(method)), asText(getDocTree(method))).write();
        }
        table().writeClose();
    }

    private void writeMethodDetails() {
        hr().write();
        for (ExecutableElement method : getRestMethods(service)) {
            writeMethodDetail(method);
        }
    }

    private void writeMethodDetail(ExecutableElement method) {

        // Method signature and comments
        anchor(getAnchor(method)).write();
        h2(getHttpMethod(method), " ", getUri(method)).cssClass("resource-header").write();
        p(asText(getDeprecatedTags(method))).write();

        String restDetails = parseRestDetails(method);

        div(asText(getDocTree(method)) + restDetails).write();

        // Permissions
        writePermissions(method);

        //Headers
        writeHeaders(method);

        // Parameters
        writeParameters(method);

        // Return type
        writeReturnType(method);

        List<? extends DocTree> seeDocTree = getSeeTags(method);
        // See tags
        for (DocTree seeTag : seeDocTree) {
            List<SeeTree> seeTreeReferenceList = (List<SeeTree>) ((SeeTree) seeTag).getReference();
            definitionList("See", asText(seeTreeReferenceList)).write();
        }
    }

    private void writeHeaders(ExecutableElement method) {
        List<DocTree> headersDocTree = getHeadersTags(method);
        com.xebialabs.commons.html.Element dt = definitionList("Headers");
        if (!headersDocTree.isEmpty()) {
            for (DocTree header : headersDocTree) {
                dt.add(element("dd", italic((firstWord(header))), restOfSentence(header)));
            }
            dt.write();
        }
    }

    private void writePermissions(ExecutableElement method) {
        List<DocTree> permissionDocTree = getPermissionTags(method);
        if (!permissionDocTree.isEmpty()) {
            com.xebialabs.commons.html.Element dt = definitionList("Permissions");
            for (DocTree permission : permissionDocTree) {
                String rest = restOfSentence(permission);
                if (!Strings.isNullOrEmpty(rest)) {
                    rest = " - " + rest;
                }
                dt.add(element("dd", code(firstWord(permission)), rest));
            }
            dt.write();
        }
    }

    private String parseRestDetails(Element element) {
        List<DocTree> restDetails = getRestDetailsTags(element);
        if (restDetails.isEmpty()) {
            return "";
        } else {
            return " " + asText(restDetails);
        }
    }

    private void writeDeprecationDetails(Element element) {
        com.xebialabs.commons.html.Element p = p(element("strong", "Deprecated:"));
        List<DocTree> deprecatedTags = getDeprecatedTags(element);
        if (!deprecatedTags.isEmpty()) {
            p.add(italic(asText(deprecatedTags)));
            p.write();
        }
    }

    private void writeReturnType(ExecutableElement method) {

        if ("void".equals(method.getReturnType().toString())) {
            definitionList("Response body", italic("Empty")).write();
        } else {
            definitionList(
                    "Response body",
                    renderType(method.getReturnType()) + getReturnTypeInfo(method),
                    "Content type: " + getMethodProduces(method)
            ).write();
        }
    }

    private String getReturnTypeInfo(ExecutableElement method) {
        List<DocTree> returnTypeDocTree = getReturnTags(method);
        if (returnTypeDocTree.isEmpty()) {
            return "";
        } else {
            return " - " + asText(returnTypeDocTree);
        }
    }

    private void writeParameters(ExecutableElement method) {
        List<? extends VariableElement> paramElements = method.getParameters();
        if (paramElements.isEmpty()) {
            return;
        }

        com.xebialabs.commons.html.Element table = table().cssClass("parameter-table");
        for (VariableElement paramElement : paramElements) { // see above
            ParameterInfo info = getParameterInfo(method, paramElement);
            if (info == null) {
                System.out.println("Warning: No actual parameter for @param " + paramElement.getSimpleName().toString() + " on " + method);
                continue;
            }
            List<DocTree> allParamTreeList = getParamTags(method);
            List<DocTree> filteredElementForRow = new ArrayList<>();
            if (!allParamTreeList.isEmpty()) {
                filteredElementForRow = allParamTreeList.stream().filter(p -> p.toString().contains(paramElement.toString())).collect(toList());
            }
            table.add(row(italic(info.kind), info.name, renderType(info.type), asText(filteredElementForRow)));
        }
        definitionList("Parameters", table).write();
    }

    //
    // Inspection
    //

    private static List<ExecutableElement> getRestMethods(TypeElement service) {
        MethodScanner scanner = new MethodScanner();
        scanner.scan(service, null);

        return scanner.getMethods().stream().filter(RestServiceWriter::isRestMethod).sorted(new MethodComparator()).collect(toList());
    }

    //
    private static boolean isRestMethod(Element element) {
        List<? extends AnnotationMirror> mirrors = element.getAnnotationMirrors();
        for (AnnotationMirror mirror : mirrors) {
            if (mirror.getAnnotationType().toString().startsWith("javax.ws.rs")) {
                return true;
            }
        }
        return false;
    }

    private static String getPath(Element element) {
        return getAnnotationValue(element, "javax.ws.rs.Path");
    }

    private String getUri(Element element) {
        return path + "/" + getPath(element);
    }

    //
    private String getAnchor(Element element) {
        return getUri(element) + ":" + getHttpMethod(element);
    }

    //
    private String getConsumes(Element element) {
        return optionallyStripJson(getAnnotationValue(element, "javax.ws.rs.Consumes").replace("\"", "").replaceAll("[\"\\]\\[]", ""));
    }

    private String optionallyStripJson(String s) {
        if (!printJsonContentType) {
            return s.replace("application/json", "").replaceAll(",\\s*$", "");
        }
        return s;
    }

    private String getMethodConsumes(ExecutableElement method) {
        String consumes = getConsumes(method);
        if (Strings.isNullOrEmpty(consumes)) {
            return defaultConsumes;
        }
        return consumes;
    }

    private String getProduces(Element element) {
        return optionallyStripJson(getAnnotationValue(element, "javax.ws.rs.Produces").replaceAll("[\"\\]\\[]", ""));
    }

    private String getMethodProduces(ExecutableElement method) {
        String produces = getProduces(method);
        if (Strings.isNullOrEmpty(produces)) {
            return defaultProduces;
        }
        return produces;
    }

    private static String getHttpMethod(Element element) {
        for (AnnotationMirror annotation : element.getAnnotationMirrors()) {
            if (annotation.getAnnotationType().toString().equals("javax.ws.rs.GET")) {
                return "GET";
            }
            if (annotation.getAnnotationType().toString().equals("javax.ws.rs.POST")) {
                return "POST";
            }
            if (annotation.getAnnotationType().toString().equals("javax.ws.rs.PUT")) {
                return "PUT";
            }
            if (annotation.getAnnotationType().toString().equals("javax.ws.rs.DELETE")) {
                return "DELETE";
            }
            if (annotation.getAnnotationType().toString().equals("javax.ws.rs.HEAD")) {
                return "HEAD";
            }
        }
        return "?";
    }

    //
    public static String getAnnotationValue(Element element, String annotationType) {
        return getAnnotationValue(getAnnotation(element, annotationType));
    }

    private static String getAnnotationValue(AnnotationMirror annotation) {
        if (annotation == null) {
            return "";
        }

        for (AnnotationValue item : annotation.getElementValues().values()) {
            Object value = item.getValue();
            if (value instanceof Object[]) {
                return Arrays.asList((Object[]) value).toString();
            }
            return value.toString();
        }

        return "";
    }

    private static AnnotationMirror getAnnotation(Element element, String type) {
        return element.getAnnotationMirrors().stream().filter(mirror -> mirror.getAnnotationType().toString().equals(type)).findFirst().orElse(null);
    }

    private ParameterInfo getParameterInfo(ExecutableElement method, VariableElement tag) {

        List<? extends VariableElement> parameters = method.getParameters();
        String name = tag.getSimpleName().toString();
        for (VariableElement param : parameters) {
            if (!param.getSimpleName().toString().equals(name)) {
                continue;
            }

            TypeMirror type = param.asType();
            for (AnnotationMirror annotation : param.getAnnotationMirrors()) {
                if (annotation.getAnnotationType().toString().equals("javax.ws.rs.PathParam")) {
                    return new ParameterInfo(getAnnotationValue(annotation), "Path", type);
                }
                if (annotation.getAnnotationType().toString().equals("javax.ws.rs.QueryParam")) {
                    return new ParameterInfo(getAnnotationValue(annotation), "Query", type);
                }
                if (annotation.getAnnotationType().toString().equals("javax.ws.rs.HeaderParam")) {
                    return new ParameterInfo(getAnnotationValue(annotation), "Header", type);
                }
                if (annotation.getAnnotationType().toString().equals("org.jboss.resteasy.annotations.providers.multipart.MultipartForm")) {
                    return new ParameterInfo(getAnnotationValue(annotation), "Multipart", type);
                }
            }
            return new ParameterInfo(getMethodConsumes(method), "Request&nbsp;body", type);
        }

        return null;
    }

    private static class ParameterInfo {
        final String name;
        final String kind;
        final TypeMirror type;

        ParameterInfo(String name, String kind, TypeMirror type) {
            this.name = name;
            this.kind = kind;
            this.type = type;
        }
    }

    private static class MethodComparator implements Comparator<ExecutableElement> {

        @Override
        public int compare(ExecutableElement method, ExecutableElement anotherMethod) {
            return getPath(method).compareTo(getPath(anotherMethod));
        }
    }
}
