package com.xebialabs.gradle.documentation.restdoc.doclet;

import com.google.common.base.Strings;
import com.sun.javadoc.*;
import com.xebialabs.commons.html.HtmlWriter;

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * HtmlWriter with convenience methods for writing REST documentation.
 */
public class RestdocWriter extends HtmlWriter {

    private final FileCatalog fileCatalog;

    public RestdocWriter(PrintWriter writer) {
        super(writer);
        fileCatalog = FileCatalog.SINGLETON;
    }

    protected String asText(Tag[] tags) {
        StringBuilder builder = new StringBuilder();
        for (Tag tag : tags) {
            if (tag instanceof SeeTag) {
                appendLink(builder, (SeeTag) tag);
            } else if (tag.name().equals("@code")) {
                builder.append(code(tag.text()));
            } else if (tag.name().equals("@deprecated")) {
                builder.append(span("(Deprecated)").cssClass("deprecated"));
            } else {
                builder.append(tag.text());
            }
        }
        return builder.toString();
    }

    private void appendLink(StringBuilder builder, SeeTag tag) {
        String file = RestDoclet.fileNameFor(tag.referencedClassName());
        String text = tag.text();
        if (!Strings.isNullOrEmpty(tag.label())) {
            text = tag.label();
        }

        if (file != null && fileCatalog.check(file)) {
            builder.append(link(file, text));
        } else {
            builder.append(bold(text));
        }
    }

    protected String asText(Type type) {
        StringBuilder builder = new StringBuilder();
        builder.append(type.simpleTypeName());

        String separator = " of ";
        for (Type paramType : getParametrizedTypes(type)) {
            builder.append(separator);
            builder.append(paramType.simpleTypeName());
            separator = ", ";
        }

        return builder.toString();
    }

    public static String firstWord(Tag tag) {
        return tag.text().split("\\s")[0];
    }

    public static String restOfSentence(Tag tag) {
        return tag.text().substring(firstWord(tag).length());
    }

    protected String asReference(Type type) {
        List<Type> types = getParametrizedTypes(type);

        // Default case: fully qualified name of non-parametrized type.
        if (types.isEmpty()) {
            return type.qualifiedTypeName() + ".html";
        }

        // Cook something up for 'List of'
        return types.get(0) + "-" + type.simpleTypeName() + ".html";
    }

    public List<Type> getParametrizedTypes(Type type) {

        List<Type> types = new ArrayList<>();

        ParameterizedType paramType = type.asParameterizedType();
        if (paramType != null) {
            Collections.addAll(types, paramType.typeArguments());
        }

        return types;
    }

    protected Object renderType(Type type) {
        Object returnTypeText = asText(type);
        String externalFile = asReference(type);

        // Add references to catalog
        if (fileCatalog.check(externalFile)) {
            returnTypeText = link(externalFile, returnTypeText);
        }
        for (Type paramType : getParametrizedTypes(type)) {
            fileCatalog.check(asReference(paramType));
        }

        return returnTypeText;
    }

    protected Tag[] getDeprecatedTags(Doc service) {
        return service.tags("deprecated");
    }

    protected Tag[] getTags(Doc service) {
        Tag[] deprecatedTags = getDeprecatedTags(service);
        Tag[] firstSentenceTags = service.firstSentenceTags();
        if (deprecatedTags.length > 0) {
            Tag[] tags = Arrays.copyOf(firstSentenceTags, firstSentenceTags.length + 1);
            tags[firstSentenceTags.length] = deprecatedTags[0];
            return tags;
        }
        return firstSentenceTags;
    }
}
