package com.xebialabs.gradle.documentation.restdoc

import org.gradle.api.GradleException
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.plugins.JavaBasePlugin
import org.gradle.api.tasks.TaskProvider
import org.gradle.api.tasks.bundling.Zip
import org.gradle.api.tasks.compile.AbstractCompile
import org.gradle.api.tasks.javadoc.Javadoc

class RestdocletPlugin implements Plugin<Project> {
  public static final DOCUMENTATION_CONFIGURATION = "documentation"
  public static final String RESTDOC_TASK_NAME = "restdoc"

  @Override
  void apply(Project project) {
    if (!project.plugins.hasPlugin("java")) {
      throw new GradleException("Java plugin not applied, cannot apply restdoc plugin.")
    }

    def restDocExtension = project.extensions.create("restdoc", RestDocletExtension)

    project.configurations.maybeCreate(DOCUMENTATION_CONFIGURATION)
    TaskProvider<Javadoc> restDocTaskProvider = project.tasks.register(RESTDOC_TASK_NAME, Javadoc)
    restDocTaskProvider.configure {
      group = JavaBasePlugin.DOCUMENTATION_GROUP
      description = 'Generate REST API documentation'
      dependsOn('processResources')
      destinationDir = project.file("${project.buildDir}/docs/rest-api")

    }
    // rest of configuration is in afterEvaluate:
    //  fetching classpath will resolve dependencies and it will not be possible to add dependencies later
    project.gradle.taskGraph.whenReady { taskGraph ->
      // Need to configure the classpath/docletpath in whenReady so things will work
      // with the release plugin. If we do this too early the upstream jar will not have
      // the right uploadTaskName (no version, since that is set when the task graph is ready)
      restDocTaskProvider.configure {
        Javadoc restDocTask = it
        source project.sourceSets.main.allJava

        classpath = project.sourceSets.main.output + project.sourceSets.main.compileClasspath

        options { options ->
          def dlp = project.rootProject.buildscript.configurations.classpath + restDocTask.classpath
          logger.debug "restdoc task classpath: ${restDocTask.classpath.files}"
          logger.debug "docletpath: ${dlp.files}"
          options.doclet = "com.xebialabs.gradle.documentation.restdoc.doclet.RestDoclet"
          options.docletpath = dlp.files as List
          options.addStringOption("showJsonSupport", "${-> restDocExtension.printJson}")
          options.addStringOption("product", "${-> restDocExtension.productName}")
          options.addStringOption("jsonOutputFile", "${-> project.name}.json")
        }
      }
    }

    project.tasks.withType(AbstractCompile).each { t ->
      restDocTaskProvider.configure {
        dependsOn(t)
      }
    }

    project.tasks.named("jar").configure {
      dependsOn(RESTDOC_TASK_NAME)
    }

    def zipTask = project.tasks.register("restdocZip", Zip) {
      dependsOn("restdoc")
      classifier = "restdoc"
      from "${project.buildDir}/docs/rest-api"
      into "rest-api"
    }

    project.artifacts.add(DOCUMENTATION_CONFIGURATION, zipTask)
  }
}
