package com.xebialabs.gradle.plugins.restdoclet.doclet

import com.sun.javadoc.{ClassDoc, MethodDoc}

import scala.annotation.tailrec

object JythonDocUtils {

  /**
    * Traverses up to the ancestors of the class in order to find and return first non-empty
    * method doc.
    */
  private[doclet] def closestDocumented(m: MethodDoc): MethodDoc = {

    def compareMethods(m1: MethodDoc, m2: MethodDoc): Boolean = {
      m1.name() == m2.name() &&
        m1.parameters().length == m2.parameters().length &&
        m1.parameters().map(Option.apply)
          .zipAll(m2.parameters().map(Option.apply), None, None).forall {
          case (Some(candidateParam), Some(methodParam)) =>
            candidateParam.typeName() == methodParam.typeName() &&
              candidateParam.name() == methodParam.name()
          case _ => false
        }
    }

    (m.tags() ++ m.inlineTags()).toList match {
      case Nil =>
        m.containingClass().interfaces().flatMap(_.methods())
          .find(candidateMethodDoc => compareMethods(candidateMethodDoc, m))
          .getOrElse(m)
      case _ => m
    }
  }

  private[doclet] def exposedMethods(c: ClassDoc): Seq[MethodDoc] = exposedMethods(Seq(c))

  @tailrec
  private def exposedMethods(cc: Seq[ClassDoc], initial: Seq[MethodDoc] = Seq()): Seq[MethodDoc] = {

    val ownMethods = cc.flatMap(_.methods().filter(m => m.isPublic && !m.isStatic))

    val ancestors = cc.flatMap {
      c => c.interfaces() ++ Option(c.superclass()).filterNot(sc => Seq("java.lang.Object", "java.lang.Enum").contains(sc.qualifiedTypeName()))
    }

    val notOverridenOwn = ownMethods.filterNot(om => initial.exists {
      case i: MethodDoc => i.compareTo(om) == 0
    })
    ancestors match {
      case Nil => initial ++ notOverridenOwn
      case _ => exposedMethods(ancestors, initial ++ notOverridenOwn)
    }

  }

}
