package com.xebialabs.gradle.plugins.restdoclet.doclet

import com.sun.javadoc._
import spray.json._

trait DocFormats extends CollectionFormats with StandardFormats with AdditionalFormats {

  implicit val methodDocFormat = new MethodDocFormat
  implicit val classDocFormat = new ClassDocFormat
  implicit val parameterFormat = new ParameterFormat
  implicit val fieldDocFormat = new FieldDocFormat
  implicit val constructorDocFormat = new ConstructorDocFormat

  private def firstTagText(doc: Doc, tt: Array[Tag]): JsValue = tt.headOption
    .map(returnTag => JsString(TagUtils.textFromTags(doc, returnTag.inlineTags())))
    .getOrElse(JsNull)

  private def deprecationText(method: ExecutableMemberDoc): JsValue =
    firstTagText(method, method.tags("@deprecated")) match {
      case JsNull => method.annotations().find(_.annotationType().qualifiedName() == "java.lang.Deprecated").map(any => JsString("")).getOrElse(JsNull)
      case nonNull => nonNull
    }

  private def apiDetailsText(doc: Doc): String = {
    doc.tags("@apiDetails").headOption
      .map(tag => TagUtils.textFromTags(doc, tag.inlineTags())) match {
      case Some(s: String) if !s.isEmpty => " " + s
      case _ => ""
    }
  }

  private def getType(t: Type): String = {
    Option(t.asParameterizedType()) match {
      case None => t.toString
      case Some(pt) => pt.toString
    }
  }

  private def paramsWithTags(method: ExecutableMemberDoc): Seq[ParamWithTag] = {
    // Equalizing amount of param tags and params
    val paramTags = method.paramTags().map(Some(_)) ++ List.fill(method.parameters().length - method.paramTags().length)(None)
    method.parameters().zip(paramTags)
  }

  class ConstructorDocFormat extends JsonFormat[ConstructorDoc] {
    override def write(constructor: ConstructorDoc): JsValue = {
      JsObject(Map(
        "description" -> JsString(TagUtils.textFromTags(constructor, constructor.inlineTags()) + apiDetailsText(constructor)),
        "parameters" -> paramsWithTags(constructor).toJson,
        "deprecated" -> deprecationText(constructor),
        "annotations" -> JsArray(constructor.annotations().map(a => JsString(a.annotationType().qualifiedTypeName())).toList)
      ))
    }

    override def read(json: JsValue): ConstructorDoc = ???
  }

  class MethodDocFormat extends JsonFormat[MethodDoc] {

    override def write(method: MethodDoc): JsValue = {

      val methodGeneric = method.typeParameters().toSeq match {
        case Nil => JsNull
        case params => JsString(s"<${params.map(_.typeName()).mkString(", ")}>")
      }

      val documentedMethod = JythonDocUtils.closestDocumented(method)

      JsObject(Map(
        "name" -> JsString(method.name()),
        "parameters" -> paramsWithTags(documentedMethod).toJson,
        "methodGeneric" -> methodGeneric,
        "returnType" -> JsString(getType(method.returnType())),
        "description" -> JsString(TagUtils.textFromTags(method, documentedMethod.inlineTags()) + apiDetailsText(method)),
        "returnDescription" -> firstTagText(method, documentedMethod.tags("@return")),
        "deprecated" -> deprecationText(method),
        "permission" -> firstTagText(documentedMethod, documentedMethod.tags("@permission")),
        "header" -> firstTagText(documentedMethod, documentedMethod.tags("@headers")),
        "annotations" -> JsArray(method.annotations().map(a => JsString(a.annotationType().qualifiedTypeName())).toList)
      ))
    }

    override def read(json: JsValue): MethodDoc = ???
  }

  class ClassDocFormat extends JsonFormat[ClassDoc] {

    override def write(`class`: ClassDoc): JsValue = {
      JsObject(Map(
        "name" -> JsString(`class`.qualifiedName()),
        "description" -> JsString(TagUtils.textFromTags(`class`, `class`.inlineTags()) + apiDetailsText(`class`)),
        "methods" -> JythonDocUtils.exposedMethods(`class`).toJson,
        "constructors" -> `class`.constructors().toJson,
        "fields" -> `class`.fields().toList.filter(_.isPublic).toJson,
        "deprecated" -> firstTagText(`class`, `class`.tags("@deprecated")),
        "annotations" -> JsArray(`class`.annotations().map(a => JsString(a.annotationType().qualifiedTypeName())).toList),
        "constants" -> JsArray(`class`.enumConstants().toList.map(c => JsString(c.name()))),
        "superClass" -> getSuperClassName(`class`)
      ))
    }

    override def read(json: JsValue): ClassDoc = ???

    private def getSuperClassName(`class`: ClassDoc): JsValue = {
      Option(`class`.superclassType()) match {
        case Some(t) => JsString(t.qualifiedTypeName())
        case None => JsNull
      }
    }
  }

  class FieldDocFormat extends JsonFormat[FieldDoc] {

    override def write(field: FieldDoc): JsValue = {
      JsObject(Map(
        "name" -> JsString(field.name()),
        "type" -> JsString(getType(field.`type`())),
        "description" -> JsString(TagUtils.textFromTags(field, field.inlineTags())),
        "annotations" -> JsArray(field.annotations().map(a => JsString(a.annotationType().qualifiedTypeName())).toList)
      ))
    }

    override def read(json: JsValue): FieldDoc = ???
  }

  type ParamWithTag = (Parameter, Option[ParamTag])

  class ParameterFormat extends JsonFormat[ParamWithTag] {

    override def write(obj: ParamWithTag): JsValue = {
      JsObject(Map(
        "name" -> JsString(obj._1.name()),
        "type" -> JsString(getType(obj._1.`type`())),
        "comment" -> obj._2.map((pt: ParamTag) => JsString(pt.parameterComment())).getOrElse(JsNull)
      ))
    }

    override def read(json: JsValue): ParamWithTag = ???
  }

}
