package com.xebialabs.gradle.documentation.restdoc.doclet;

import com.google.common.base.Preconditions;
import com.sun.javadoc.*;
import com.xebialabs.gradle.plugins.restdoclet.doclet.DocletUtils;

import java.io.*;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;

/**
 * Doclet implementation for generating REST documentation. Main entry point for the Javadoc tool.
 */
public class RestDoclet {
    private static File destinationDir = new File(".");
    private static boolean showJsonSupport = true;
    private static String productName = null;
    private static String jsonOutputFile = "api.json";

    /**
     * Needed by Javadoc engine.
     */
    @SuppressWarnings("unused")
    public static int optionLength(String option) {
        // Dummy implementation
        return 2;
    }

    /**
     * Needed by Javadoc engine.
     */
    @SuppressWarnings("unused")
    public static boolean validOptions(String[][] options, DocErrorReporter reporter) {
        for (String[] option : options) {
            switch (option[0]) {
                case "-d":
                    destinationDir = new File(option[1]);
                    break;
                case "-showJsonSupport":
                    showJsonSupport = Boolean.parseBoolean(option[1]);
                    break;
                case "-product":
                    productName = option[1];
                    break;
                case "-jsonOutputFile":
                    jsonOutputFile = option[1];
                    break;
            }
        }
        return true;
    }

    /**
     * Needed by Javadoc engine to provide generic type information.
     */
    @SuppressWarnings("unused")
    public static LanguageVersion languageVersion() {
        return LanguageVersion.JAVA_1_5;
    }

    /**
     * Main entry point for Javadoc.
     */
    @SuppressWarnings("unused")
    public static boolean start(RootDoc root) {
        List<ClassDoc> apiServices = DocletUtils.findAnnotatedClasses(root, "com.xebialabs.xlplatform.documentation.PublicApi");

        List<ClassDoc> apiRefs = DocletUtils.findAnnotatedClasses(root, "com.xebialabs.xlplatform.documentation.PublicApiRef");

        List<ClassDoc> allApiClasses = newArrayList(apiServices);
        allApiClasses.addAll(apiRefs);

        Preconditions.checkNotNull(productName, "The productName should not be 'null', please specify '-product'");

        if (!apiServices.isEmpty()) {
            copyCss();
            writeOverview(apiServices);
            writeServices(apiServices);
            writeLinkedFiles();
        }
        writeJsonFile(allApiClasses);

        return true;
    }

    private static void writeJsonFile(List<ClassDoc> restServices) {
        File destination = new File(destinationDir, jsonOutputFile);
        try (FileWriter fileWriter = new FileWriter(destination)) {
            fileWriter.write(DocletUtils.generateJson(restServices));
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    private static void copyCss() {
        Resource.fromClasspath("restdoc/layout.css").copy(new File(destinationDir, "layout.css"));
        Resource.fromClasspath("restdoc/restdoc.css").copy(new File(destinationDir, "restdoc.css"));
        Resource.fromClasspath("restdoc/image.zip").unzip(destinationDir);
    }


    private static void writeOverview(List<ClassDoc> restServices) {
        PrintWriter writer = openFile("index.html", String.format("%s REST API", productName));
        try {
            new OverviewWriter(writer, productName).write(restServices);
        } finally {
            closeFile(writer);
        }
    }

    private static void writeServices(List<ClassDoc> restServices) {
        // Register files that are going to be generated, for cross-referencing
        for (ClassDoc service : restServices) {
            FileCatalog.SINGLETON.add(fileNameFor(service));
        }

        // Generate documentation per service
        for (ClassDoc service : restServices) {
            PrintWriter writer = openFile(fileNameFor(service), service.name());
            try {
                new RestServiceWriter(writer, service, showJsonSupport).writeRestService();
            } finally {
                closeFile(writer);
            }
        }
    }

    private static void writeLinkedFiles() {
        for (String item : FileCatalog.SINGLETON.getItems()) {
            if (FileCatalog.isResource(item)) {
                PrintWriter writer = openFile(item, item.replace(".html", ""));
                try {
                    FileCatalog.write(item, writer);
                } finally {
                    closeFile(writer);
                }
            }
        }

        for (String item : FileCatalog.SINGLETON.getMissing()) {
            System.out.println("Missing cross reference: " + item);
        }
    }

    public static String fileNameFor(Type service) {
        if (service == null) {
            return null;
        }
        return service.qualifiedTypeName() + ".html";
    }

    public static String fileNameFor(String reference) {
        if (reference == null) {
            return null;
        }
        return reference + ".html";
    }

    public static PrintWriter openFile(String fileName, String title) {
        File file = new File(destinationDir, fileName);
        try {
            PrintWriter writer = new PrintWriter(file);
            new PageTemplate(writer).writeHeader(title);
            return writer;
        } catch (FileNotFoundException e) {
            throw new IllegalStateException("Can't open " + file.getAbsolutePath() + " for writing.");
        }
    }

    public static void closeFile(PrintWriter writer) {
        new PageTemplate(writer).writeFooter();
        writer.flush();
        writer.close();
    }

}
