package com.xebialabs.deployit.documentation

import com.google.common.collect.ForwardingMap
import com.google.common.io.ByteSource
import com.google.common.io.ByteStreams
import com.google.common.io.Files
import com.google.common.io.LineProcessor
import com.samskivert.mustache.Mustache
import com.samskivert.mustache.MustacheException
import org.apache.hc.client5.http.classic.methods.HttpGet
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient
import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse
import org.apache.hc.client5.http.impl.classic.HttpClients
import org.slf4j.Logger
import org.slf4j.LoggerFactory

import java.nio.charset.Charset
import java.util.regex.Matcher
import java.util.regex.Pattern
import java.util.zip.ZipEntry
import java.util.zip.ZipInputStream

import static com.google.common.collect.Maps.newHashMap

class IOUtils {

  private static ByteSource newUrlInputSupplier(final URL url) {
    return new ByteSource() {
      @Override
      InputStream openStream() throws IOException {
        if (url.getProtocol().equals("file")) {
          return url.openStream();
        }
        CloseableHttpClient httpClient = HttpClients.createDefault();
        try {
          HttpGet request = new HttpGet(url.toURI());
          CloseableHttpResponse response = httpClient.execute(request);
          return response.getEntity().getContent();
        } catch (URISyntaxException e) {
          throw new IOException(e);
        }
      }
    };
  }

  static void copy(URL fromUrl, File toFile) {
    try {
      newUrlInputSupplier(fromUrl).copyTo(Files.asByteSink(toFile));
    } catch (IOException e) {
      throw new RuntimeIOException(e);
    }
  }

  static String getText(URL url) {
    try {
      return new String(newUrlInputSupplier(url).read());
    } catch (IOException e) {
      throw new RuntimeIOException(e);
    }
  }

  static String getText(File file) {
    try {
      return Files.toString(file, Charset.defaultCharset());
    } catch (IOException e) {
      throw new RuntimeIOException(e);
    }
  }


  static void copyDirectory(File sourceLocation, File targetLocation) {

    if (sourceLocation.isDirectory()) {
      if (!targetLocation.exists()) {
        targetLocation.mkdir();
      }

      for (String child : sourceLocation.list()) {
        copyDirectory(new File(sourceLocation, child), new File(targetLocation, child));
      }
    } else {
      try {
        Files.copy(sourceLocation, targetLocation);
      } catch (IOException e) {
        throw new RuntimeIOException(e);
      }
    }
  }

  static String replacePlaceholders(String replaceable, ContextProperties values) {
    StringWriter writer = new StringWriter();
    replacePlaceholders(new StringReader(replaceable), values, writer);
    return writer.toString();
  }

  static void replacePlaceholders(Reader replaceable, ContextProperties values, Writer writer) {
    try {
      if (values == null || values.isEmpty()) {
        $copy(replaceable, writer);
      }

      Mustache.compiler().compile(replaceable).execute(new MustacheContext((Map) values), writer);
    } catch (MustacheException me) {
      throw new RuntimeException("Could not replace keys in " + replaceable, me);
    } catch (IOException e) {
      throw new RuntimeIOException(e);
    }
  }

  static String getTextWithInclusions(File markdownFile) {
    Stack<File> stack = new Stack<>();
    stack.push(markdownFile);
    return $getTextWithInclusions(markdownFile, stack);
  }

  private static String $getTextWithInclusions(final File markdownFile, final Stack<File> stack) {
    LineProcessor<String> out = new LineProcessor<String>() {

      Pattern includePattern = Pattern.compile("^include::(.*)\$");
      StringBuilder sb = new StringBuilder();

      @Override
      boolean processLine(String line) throws IOException {
        Matcher m = includePattern.matcher(line);
        if (m.matches()) {
          File file = new File(markdownFile.getParent(), m.group(1));
          if (!file.exists()) {
            throw new RuntimeIOException("Could not find included file: " + file.getAbsolutePath());
          }
          if (stack.contains(file)) {
            throw new RuntimeIOException("Dependency cycle detected in file: " + file.getAbsolutePath());
          }
          stack.push(file);
          sb.append($getTextWithInclusions(file, stack)).append('\n');
          stack.pop();
        } else {
          sb.append(line).append('\n');
        }
        return true;
      }

      @Override
      String getResult() {
        return sb.toString();
      }
    };
    try {
      Files.readLines(markdownFile, Charset.defaultCharset(), out);
      return out.getResult();
    } catch (IOException e) {
      throw new RuntimeIOException(e);
    }
  }

  private static int $copy(Reader input, Writer output) throws IOException {
    char[] buffer = new char[1024 * 4];
    int count = 0;
    int n = 0;
    while (-1 != (n = input.read(buffer))) {
      output.write(buffer, 0, n);
      count += n;
    }
    return count;
  }

  public static File createUniqueFileNameWithNewExtension(File source, File workingDir, String ext) {
    String name = extractFileNameWithoutExtension(source);
    String newFileName = name + "." + ext;
    for (int i = 1; i < 50; i++) {
      File newFile = new File(workingDir, newFileName);
      if (!newFile.exists()) {
        return newFile;
      }
      newFileName = name + i + "." + ext;
    }
    throw new RuntimeException("Cannot generate unique file name for '" + name + "." + ext + "' in directory " + workingDir.getAbsolutePath());
  }

  public static File createFileNameWithNewExtension(File source, File workingDir, String ext) {
    String name = extractFileNameWithoutExtension(source);
    String newFileName = name + "." + ext;
    File newFile = new File(workingDir, newFileName);
    if (newFile.exists()) {
      newFile.delete();
      try {
        newFile.createNewFile();
      } catch (IOException e) {
        throw new RuntimeIOException();
      }
    }
    return newFile;
  }

  public static String extractFileNameWithoutExtension(File file) {
    String fileName = file.getName();
    int index = fileName.lastIndexOf('.');
    if (index > -1) {
      fileName = fileName.substring(0, index);
    }
    return fileName;
  }

  public static String extractFileName(URL url) {
    String urlAsString = url.toString();
    if (urlAsString.endsWith("/")) {
      urlAsString = urlAsString.substring(0, urlAsString.length() - 1);
    }

    int slashIndex = urlAsString.lastIndexOf('/');
    return urlAsString.substring(slashIndex + 1);
  }

  public static String extractFileNameExtension(String fileName) {
    int index = fileName.lastIndexOf('.');
    if (index > -1) {
      return fileName.substring(index + 1);
    }
    return "";
  }

  public static File explodeArchive(File archive, File explodedFolder) {
    try {
      final ZipInputStream zipEntryStream = new ZipInputStream(new FileInputStream(archive));
      if (!explodedFolder.exists()) {
        explodedFolder.mkdir();
      }
      for (; ;) {
        ZipEntry entry = zipEntryStream.getNextEntry();
        if (entry == null) {
          return explodedFolder;
        }

        try {

          if (entry.isDirectory()) {
            final File file = new File(explodedFolder, entry.getName());
            if (!file.exists()) {
              isTrue(file.mkdirs(), "Could not create directory: " + entry.getName());
            }
            continue;
          }

          final File destinationFile = new File(explodedFolder, entry.getName());
          final String parentDirPath = destinationFile.getParent();
          if (parentDirPath != null) {
            final File destinationDir = new File(parentDirPath);
            if (!destinationDir.exists()) {
              isTrue(destinationDir.mkdirs(), "Could not create directory: " + entry.getName());
            }
          }
          if (!destinationFile.exists())
            isTrue(destinationFile.createNewFile(), "Could not create file: " + entry.getName());
          ByteStreams.copy(zipEntryStream, new FileOutputStream(destinationFile));
        } finally {
          zipEntryStream.closeEntry();
        }
      }
    } catch (IOException exc) {
      // On exception, clean up!
      try {
        deleteRecursively(explodedFolder);
      } catch (Exception e) {
        logger.error("Could not delete {}", explodedFolder, e);
      }
      throw new RuntimeIOException(exc);
    }
  }

  public static boolean deleteRecursively(File path) throws FileNotFoundException {
    if (!path.exists()) throw new FileNotFoundException(path.getAbsolutePath());
    boolean ret = true;
    if (path.isDirectory()) {
      for (File f : path.listFiles()) {
        ret = ret && deleteRecursively(f);
      }
    }
    return ret && path.delete();
  }

  private static void isTrue(boolean expression, String message) {
    if (!expression) {
      throw new IllegalArgumentException(message);
    }
  }

  private static class MustacheContext extends ForwardingMap<String, Object> {

    private Map<String, Object> delegate = newHashMap();

    private MustacheContext(Map<String, Object> delegate) {
      this.delegate = delegate;
    }

    @Override
    protected Map<String, Object> delegate() {
      return delegate;
    }

    @Override
    public Object get(Object key) {
      if (delegate.containsKey(key)) {
        return delegate.get(key);
      }

      String prefix = key + ".";
      Map<String, Object> map = newHashMap();
      for (String fullKey : delegate.keySet()) {
        if (fullKey.startsWith(prefix)) {
          map.put(fullKey.substring(prefix.length()), delegate.get(fullKey));
        }
        if (!map.isEmpty()) {
          return new MustacheContext(map);
        }
      }

      return null;
    }

  }

  private static final Logger logger = LoggerFactory.getLogger(IOUtils.class);
}
