package com.xebialabs.gradle.plugins.docgen.tasks

import com.xebialabs.gradle.plugins.docgen.DocumentGeneratorPlugin
import org.gradle.api.DefaultTask
import org.gradle.api.Project
import org.gradle.api.artifacts.ProjectDependency
import org.gradle.api.plugins.JavaBasePlugin
import org.gradle.api.tasks.CacheableTask
import org.gradle.api.tasks.InputDirectory
import org.gradle.api.tasks.OutputDirectory
import org.gradle.api.tasks.PathSensitive
import org.gradle.api.tasks.PathSensitivity
import org.gradle.api.tasks.TaskAction

import static com.xebialabs.gradle.plugins.docgen.DocumentGeneratorPlugin.DOC_SOURCE_CONFIGURATION

@CacheableTask
class GenerateDocumentation extends DefaultTask {

  @OutputDirectory
  File outputDir

  @InputDirectory
  @PathSensitive(PathSensitivity.ABSOLUTE)
  File srcDir

  private transient GenerateDocumentationExtension taskext = project.extensions.findByName("generateDocumentation")

  GenerateDocumentation() {
    group = JavaBasePlugin.DOCUMENTATION_GROUP
    description = "Generate the ${project.name} documentation."
  }

  @TaskAction
  public void generateDocumentation() {
    registerProjectWithExtension(getProject())
    project.configurations.getByName(DOC_SOURCE_CONFIGURATION).dependencies.withType(ProjectDependency).each { projectDep ->
      registerProjectWithExtension(projectDep.dependencyProject)
    }

    taskext.config.put("version", (taskext.version != null) ? taskext.version : project.version)

    def propertyFile = project.file("${temporaryDir}/props.properties")
    writeProperties(propertyFile)

    def cmdArgs = getCommandArgs(propertyFile)

    // the DocumentGenenerator uses LocalBooter (CI's) and scannit (steps) so we need a classpath that
    // has all the needed stuff in there. To prevent issues with other LocalBooter bootups in the same VM it's
    // better to do the execution via a javaexec step then to boot the LocalBooter in gradle space, that might
    // leave static stuff around in a daemon.

    def docgenCP
    if (project.hasProperty('docgenClasspathJar')) {
      docgenCP = project.files(project.tasks['docgenClasspathJar'])
    } else {
      DocumentGeneratorPlugin.addDocClasspath(project, taskext.docClasspath)
      docgenCP = project.files(taskext.docClasspath)
    }

    logger.info("Going to invoke Document Generator with args: $cmdArgs")
    logger.info("Going to invoke Document Generator with classpath: $docgenCP")

    project.javaexec {
      classpath = docgenCP
      setMain 'com.xebialabs.deployit.documentation.DocumentGenerator'
      args = cmdArgs
      jvmArgs = ['-Djdk.module.illegalAccess=deny']
    }
  }

  def writeProperties(File propertyFile) {
    taskext.config.toProperties().store(propertyFile.newWriter(), "")
  }

  def getCommandArgs(File propertyFile) {
    def args = ['-workingDir', outputDir,
                '-htmlTemplate', asResource(taskext.htmlTemplate),
                '-properties', propertyFile.absolutePath,
                '-htmlHeader', asResource(taskext.htmlHeader),
                '-htmlFooter', asResource(taskext.htmlFooter)]

    taskext.docResources.each { args << "-resource" << asResource(it) }
    project.fileTree(srcDir).include("*.markdown").each { args << "-source" << "${it}" }
    logger.info("${args}")
    return args
  }

  def asResource(String r) {
    return project.file("${project.buildDir}/docgen-resources/${r}").toURI()
  }

  def registerProjectWithExtension(Project docProject) {
    if (docProject.hasProperty("docgen")) {
      taskext.config.putAll(docProject.getProperty("docgen"))
    }

    if (project == docProject) {
      taskext.docClasspath.add(project.file("${project.buildDir}/classes/java/main"))
      taskext.docClasspath.add(project.file("${project.buildDir}/classes/scala/main"))
      taskext.docClasspath.add(project.file("${project.buildDir}/resources/main"))
    } else {
      taskext.docClasspath.addAll(docProject.configurations.getByName("archives").allArtifacts.files.filter { f ->
        !f.name.contains("-sources") && !f.name.contains("-javadoc")
      }.files)
      taskext.docClasspath.addAll(docProject.configurations.getByName("compile").resolve())
    }
  }
}
