package com.xebialabs.gradle.plugins.docgen

import com.xebialabs.deployit.documentation.DocumentGenerator
import com.xebialabs.gradle.plugins.docgen.tasks.GenerateDocumentation
import com.xebialabs.gradle.plugins.docgen.tasks.GenerateDocumentationExtension
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.plugins.JavaBasePlugin
import org.gradle.api.tasks.bundling.Jar
import org.gradle.api.tasks.compile.JavaCompile
import org.slf4j.Logger
import org.slf4j.LoggerFactory

class DocumentGeneratorPlugin implements Plugin<Project> {
  private static final Logger log = LoggerFactory.getLogger(DocumentGeneratorPlugin.class)

  public static final String DOC_GENERATOR_CONFIGURATION = "docgenerator"
  public static final String DOC_SOURCE_CONFIGURATION = "docgen"

  GenerateDocumentationExtension extension
  Project project

  boolean isApplied = false

  static String shortVersion(def version) {
    version.toString().replaceAll(/^(\d+)\.(\d+)(?:\.\d+.*)$/, { all, major, minor ->
      "${major}.${minor}.x"
    })
  }

  @Override
  void apply(Project _project) {
    this.project = _project

    getClass().getResourceAsStream('/gradle-doc-gen-internal.properties').withCloseable { is ->
      def props = new Properties()
      props.load(is)
      props.each { prop ->
        project.dependencies.ext.set("${prop.key}DocGenInternal", prop.value)
      }
    }

    project.rootProject.plugins.apply("base")

    def srcDir = project.file("${project.projectDir}/src/main/markdown")
    log.info("DocumentGeneratorPlugin srcDir={}", srcDir)

    if (!srcDir.exists()) return

    isApplied = true

    project.ext.docgen = [
        year: new Date()[Calendar.YEAR],
        realVersion: project.version.toString(),
        shortVersion: shortVersion(project.version),
    ]

    log.debug("DocumentGeneratorPlugin adding configurations")

    project.configurations.create("docbasestyle")
    project.configurations.create(DOC_SOURCE_CONFIGURATION)
    project.configurations.create(DOC_GENERATOR_CONFIGURATION)

    project.dependencies {
      // explicitly load gradle api dependencies
      docgenerator gradleApi()

      // let these versions come from dependency plugin, especially important for udm-plugin-api
      docgenerator "com.xebialabs.deployit.engine:local-booter"
      docgenerator "com.xebialabs.deployit:udm-plugin-api"

      // to be able to run the java DocumentGenerator code that it's embedded in the plugin jar
      // we need to have some dependencies. This configuration takes care of that.
      //
      // The dependencies are likely to get managed by the dependency plugin, but like this
      // the plugin will not fail if a key gets removed from the reference
      // This should be kept in sync with the compile configuration in the build.gradle
      docgenerator "args4j:args4j:$args4jVersionDocGenInternal"
      docgenerator "com.google.guava:guava:$guavaVersionDocGenInternal"
      docgenerator "com.samskivert:jmustache:$jmustacheVersionDocGenInternal"
      docgenerator "nl.javadude.scannit:scannit:$scannitVersionDocGenInternal"
      docgenerator "org.apache.httpcomponents:httpclient:$httpClientVersionDocGenInternal"
      docgenerator "org.apache.httpcomponents:httpcore:$httpCoreVersionDocGenInternal"
      docgenerator("com.vladsch.flexmark:flexmark-all:$flexmarkVersionDocGenInternal") {
        exclude group: "com.vladsch.flexmark", module: "flexmark-pdf-converter"
      }
      docgenerator "org.codehaus.groovy:groovy-all:$groovyVersionDocGenInternal"

      docbasestyle group: 'com.xebialabs.deployit.documentation', name: 'base-documentation-style', ext: 'zip'
    }

    extension = project.extensions.create("generateDocumentation", GenerateDocumentationExtension)
    extension.usingProject(project)

    if (project.hasProperty('docgenClasspathJar')) {
      project.tasks.create("docgenClasspathJar", Jar).with {
        archiveFile.set(new File(temporaryDir, 'docgenClasspathJar.jar'))
        doFirst {
          // the generator code is inside the plugin jar, add it
          def docClasspath = addDocClasspath(project, new LinkedHashSet<File>()).collect { it.toURI().toURL() }

          manifest {
            attributes('Main-Class': DocumentGenerator.class.getCanonicalName())
            attributes('Class-Path': docClasspath.join(' '))
          }
        }
      }
    }

    def unzipBaseStyleTask = project.tasks.create("unzipBaseStyleTask")
    unzipBaseStyleTask.configure {
      onlyIf { !project.hasProperty("skipDocumentation") }

      def outputDir = "${project.buildDir}/docgen-resources"
      outputs.dir(outputDir)
      // needs to be lazy so the dependency plugin has a chance to set the version
      inputs.file("${-> project.configurations.docbasestyle.singleFile}")

      doFirst {
        project.copy {
          from project.zipTree(project.configurations.docbasestyle.singleFile)
          into outputDir
        }
      }
    }

    if (srcDir.exists()) {
      createGenerateDocumentationTasks(srcDir, unzipBaseStyleTask)
    } else {
      log.info("DocumentGeneratorPlugin srcDir {} does not exist, doc tasks not added", srcDir)
    }
  }

  static LinkedHashSet<File> addDocClasspath(Project project, LinkedHashSet<File> docClasspath) {
    docClasspath.add(determinePluginJar());

    // the dependencies of the generator code in the plugin are in the documentgenerator configuration
    // which is resolved at the time that the plugin is used
    docClasspath.addAll(project.configurations.getByName(DOC_GENERATOR_CONFIGURATION).resolve())
    if (project.configurations.findByName('runtime') != null) {
      docClasspath.addAll(project.configurations.getByName("runtime").resolve())
    }
    docClasspath
  }

  static File determinePluginJar() {
    Class klass = DocumentGenerator.class;
    URL url = klass.getResource('/' + klass.getName().replace('.', '/') + ".class");
    // Cleanup the url: file:/a/bc!package.Class
    String path = url.path.replaceFirst(/!.+$/, '')
    path = path.replaceFirst(/file:/, '')
    if (System.getProperty("os.name").toLowerCase().startsWith("win")) {
      path = path.substring(1).replaceAll("/", "\\\\")
    }
    new File(path)
  }

  private void createGenerateDocumentationTasks(File srcDir, Task unzipBaseStyleTask) {
    def docTempDir = project.file("${project.buildDir}/doc")
    log.info("DocumentGeneratorPlugin createGenerateDocumentationTasks srcDir={}", srcDir)
    if (!project.fileTree(srcDir).include("*.markdown").collect().empty) {
      addGenerateDocumentationTask("", srcDir, docTempDir, unzipBaseStyleTask).doFirst {
        project.copy {
          into docTempDir
          from srcDir
          exclude "**/*.markdown"
        }
      }
    } else {
      srcDir.eachDir { dir ->
        if (!project.fileTree(dir).include("*.markdown").collect().empty) {
          def docDir = project.file("${docTempDir}/${dir.name}")
          addGenerateDocumentationTask(dir.name, dir, docDir, unzipBaseStyleTask).doFirst {
            project.copy {
              into docDir
              from dir
              exclude "**/*.markdown"
            }
          }
        }
      }
    }
  }

  static def initialCap(String s) {
    return s.empty ? s : s.substring(0, 1).toUpperCase() + s.substring(1)
  }

  def addGenerateDocumentationTask(String which, File sourceDir, File workingDir, Task unzipBaseStyleTask) {
    def generateTask = project.tasks.create("generate${initialCap(which)}Documentation", GenerateDocumentation)
    log.debug("adding task {}", generateTask)
    generateTask.configure { t ->
      group = JavaBasePlugin.DOCUMENTATION_GROUP
      description = "Generate documentation for ${project.name}"

      dependsOn unzipBaseStyleTask
      dependsOn project.tasks.withType(Jar)

      srcDir = sourceDir
      outputDir = workingDir

      inputs.property(DOC_SOURCE_CONFIGURATION, project.ext.docgen)
      inputs.files(unzipBaseStyleTask.outputs.files)
      inputs.dir(which ? new File(which, sourceDir) : sourceDir)
      inputs.files project.tasks.withType(JavaCompile)*.inputs.files
      outputs.dir workingDir

      onlyIf { !project.hasProperty("skipDocumentation") }
    }
    return generateTask
  }
}
