package com.xebialabs.gradle.plugins.docgen.reference.step

import com.xebialabs.deployit.plugin.api.rules.StepMetadata
import com.xebialabs.deployit.plugin.api.rules.StepParameter
import com.xebialabs.gradle.plugins.docgen.reference.step.model.Parameter
import com.xebialabs.gradle.plugins.docgen.reference.step.model.Step
import nl.javadude.scannit.Configuration
import nl.javadude.scannit.Scannit
import nl.javadude.scannit.scanner.TypeAnnotationScanner
import org.slf4j.Logger
import org.slf4j.LoggerFactory

import java.lang.annotation.Annotation
import java.lang.reflect.Field

public class StepDocumentationScanner {
  private final static Logger logger = LoggerFactory.getLogger(StepDocumentationScanner);

  /** Return step documentation for all steps sorted on step name. */
  public static List<Step> findStepDocumentation(final String packageName, final List<File> markdownSources) {
    try {
      Configuration configuration = Configuration.config().scan(packageName).with(new TypeAnnotationScanner());
      Scannit scannit = new Scannit(configuration);

      scannit.getTypesAnnotatedWith(StepMetadata).collect({ Class<?> stepClass ->
        logger.info("Processing step class : {}", stepClass.name);

        List<Parameter> parameters = getAllFields(stepClass, StepParameter).collect({ Field field ->
          StepParameter stepParameter = field.getAnnotation(StepParameter);
          new Parameter(
              getFieldName(field, stepParameter),
              field.getType().getSimpleName(),
              stepParameter.description(),
              stepParameter.required(),
              stepParameter.calculated()
          );
        }).sort({ p1, p2 -> p1.name <=> p2.name })

        String name = stepClass.getAnnotation(StepMetadata).name();
        String stepDocumentationFileName = "step-${name}.markdown"
        File md = markdownSources.find { File md ->
          md.name == stepDocumentationFileName
        }
        if (md == null) {
          throw new IllegalArgumentException("Did not find ${stepDocumentationFileName} for step ${name} from class ${stepClass.simpleName}")
        }
        if (!md.exists()) {
          throw new IllegalArgumentException("Did not find step documentation file ${md.absolutePath} for step ${name} from class ${stepClass.simpleName}")
        }
        new Step(name, md, parameters);
      }).sort({ s1, s2 -> s1.name <=> s2.name })
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  private static String getFieldName(Field field, StepParameter stepParameter) {
    String parameterName = stepParameter.name().isEmpty() ? field.name : stepParameter.name();
    toSnakeCase(parameterName)
  }

  private static String toSnakeCase(String text) {
    text.replaceAll(/([A-Z])/, /-$1/).toLowerCase()
  }

  private static List<Field> getAllFields(Class<?> aClass, Class<? extends Annotation> annotation) {
    List<Field> result = new ArrayList<Field>();

    for (Field field : aClass.getDeclaredFields()) {
      if (field.getAnnotation(annotation) != null) {
        result.add(field);
      }
    }

    Class<?> superclass = aClass.getSuperclass();
    if (superclass != null) {
      result.addAll(getAllFields(superclass, annotation));
    }

    result;
  }
}
