package ai.digital.gradle.defaults

import org.gradle.api.GradleException
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.publish.maven.MavenPublication
import org.gradle.api.publish.tasks.GenerateModuleMetadata
import org.gradle.api.tasks.compile.JavaCompile
import org.gradle.api.tasks.javadoc.Javadoc
import org.gradle.api.JavaVersion
import org.gradle.jvm.toolchain.JavaLanguageVersion

class JavaPlugin implements Plugin<Project> {
  void apply(Project project) {
    project.plugins.apply('digitalai.base')
    project.plugins.apply('java-library')
    project.plugins.apply('digitalai.artifact')
    project.plugins.apply('ru.vyarus.pom')

    project.java {
      toolchain {
        languageVersion = JavaLanguageVersion.of(11)
      }
      if (!project.hasProperty("languageLevel")) {
        throw new GradleException("Could not determine the Java language level, please add `languageLevel=(1.8|11)` to ${project.rootProject.projectDir}/gradle.properties")
      }
      def languageLevel = project.property("languageLevel")
      sourceCompatibility = languageLevel
      targetCompatibility = languageLevel
      withSourcesJar()
      withJavadocJar()
    }

    def checkJavaTask = project.rootProject.tasks.getByName('checkJavaVersion')
    project.tasks.withType(JavaCompile) { t ->
      t.dependsOn checkJavaTask
    }

    project.tasks.withType(Javadoc) {
      options.encoding = 'UTF-8'
      options.charSet = 'UTF-8'
      if (JavaVersion.current().isJava8Compatible()) {
        options.addStringOption('Xdoclint:none', '-quiet')
      }
    }

    project.pomGeneration {
      removeDependencyManagement()
    }

    project.tasks.withType(JavaCompile) {
      options.encoding = "UTF-8"
      options.incremental = true
      options.deprecation = true
    }

    project.tasks.withType(GenerateModuleMetadata) {
      enabled = false
    }

    project.publishing {
      publications {
        digitalai(MavenPublication) {
          pom {
            project.gradle.projectsEvaluated {
              name = project.description
              description = project.description
            }
            organization {
              name = 'Digital.ai'
            }
            licenses {
              license {
                name = 'EULA Artifacts XebiaLabs, Version 1.0'
                url = 'https://dist.xebialabs.com/public/legal/eula-artifacts-v10.txt'
                distribution = 'repo'
              }
            }
          }
          versionMapping {
            usage('java-api') {
              fromResolutionOf('runtimeClasspath')
            }
            usage('java-runtime') {
              fromResolutionResult()
            }
          }
          from project.components.java
        }
      }
    }
  }
}
