package ai.digital.gradle.defaults

import ai.digital.gradle.defaults.tasks.CheckJavaVersion
import ai.digital.gradle.defaults.tasks.DumpVersion
import ai.digital.gradle.defaults.tasks.NebulaPublishRelease
import ai.digital.gradle.extensions.ReleasePluginExtension
import org.gradle.api.GradleException
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.plugins.BasePlugin
import org.gradle.language.base.plugins.LifecycleBasePlugin

class RootOpinionsPlugin implements Plugin<Project> {

  private Project project

  static def EXTENSION_NAME = 'releasePlugin'

  private static void createPluginExtensions(Project project) {
    project.allprojects.each { p ->
      p.extensions.create(EXTENSION_NAME, ReleasePluginExtension)
    }
  }

  void apply(Project project) {
    assert project == project.rootProject: "Should be applied to the `rootProject` only."
    this.project = project

    project.apply plugin: "java-library"
    project.apply plugin: "nebula.release"
    project.apply plugin: "maven-publish"
    project.apply plugin: LifecycleBasePlugin

    createCheckJavaVersionTask(project)
    createPluginExtensions(project)
    createDumpVersionTask(project)
    createNebulaReleaseTask(project)
    createCheckDependencyVersionsTask(project)
  }

  private static void createNebulaReleaseTask(Project project) {
    if (project.hasProperty("nebulaBaseReleaseVersion")) {
      def nebulaReleaseTask = project.tasks.create("nebulaRelease", ai.digital.gradle.defaults.tasks.NebulaRelease) { t ->
        t.group = BasePlugin.BUILD_GROUP
        t.description = "Makes a release with pre-defined parameters"
        t.version = project.getProperties().get("nebulaBaseReleaseVersion")
      }

      def nebulaPublishReleaseTask = project.tasks.create("nebulaPublishRelease", NebulaPublishRelease) { t ->
        t.group = BasePlugin.BUILD_GROUP
        t.description = "Makes a release with pre-defined parameters"
        t.version = project.getProperties().get("nebulaBaseReleaseVersion")
      }

      def cleanTask = project.tasks.findByName("clean")
      if (cleanTask) {
        nebulaReleaseTask.mustRunAfter(cleanTask)
        nebulaPublishReleaseTask.mustRunAfter(cleanTask)
      }
    }
  }

  private static void createDumpVersionTask(Project project) {
    def dumpVersionTask = project.tasks.create("dumpVersion", DumpVersion) { t ->
      t.group = BasePlugin.BUILD_GROUP
      t.description = "Create a 'version.dump' file with the build version."
      t.conventionMapping.version = { -> project.version.toString() }
      t.dumpFile = project.file("${project.buildDir}/version.dump")
    }

    project.allprojects.each { p ->
      p.tasks.all { t ->
        if ('build' == t.name) {
          t.dependsOn dumpVersionTask
        }
      }
    }
    def cleanTask = project.tasks.findByName("clean")
    if (cleanTask) {
      dumpVersionTask.mustRunAfter(cleanTask)
    }
  }

  private static void createCheckDependencyVersionsTask(Project project) {
    def checkDeps = project.tasks.create("checkDependencyVersions", ai.digital.gradle.defaults.tasks.CheckDependencyVersions) { t ->
      t.group = LifecycleBasePlugin.VERIFICATION_GROUP
      t.description = "Ensure no dependencies have an incompatible version for the currently built project version"
      t.projects = project.allprojects
    }

    project.allprojects.each { p ->
      def processResources = p.tasks.findByName("processResources")
      if (processResources) {
        processResources.mustRunAfter(checkDeps)
      }
    }
    def prepareTask = project.tasks.getByName("prepare")
    prepareTask.dependsOn(checkDeps)
  }

  private static void createCheckJavaVersionTask(Project project) {
    if (!project.hasProperty("languageLevel")) {
      throw new GradleException("Could not determine the Java language level, please add `languageLevel=(1.8|11)` to ${project.projectDir}/gradle.properties")
    }

    project.rootProject.tasks.create("checkJavaVersion", CheckJavaVersion) { t ->
      t.group = LifecycleBasePlugin.VERIFICATION_GROUP
      t.description = "Check whether the Java version equals the required compatibility level"
      t.targetCompatibility = project.property("languageLevel")
    }
  }
}
