package com.xebialabs.gradle.defaults

import org.gradle.api.GradleException
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.plugins.JavaPlugin
import org.gradle.api.plugins.scala.ScalaBasePlugin
import org.gradle.api.tasks.compile.JavaCompile
import org.gradle.api.tasks.scala.ScalaCompile
import org.gradle.api.tasks.testing.Test

class XLOpinionsPlugin implements Plugin<Project> {

  @Override
  void apply(Project project) {
    configureRepositories(project)
    project.plugins.withType(JavaPlugin) {
      configureJava(project)
    }

    project.plugins.withType(ScalaBasePlugin) {
      configureScala(project)
    }

    project.tasks.withType(Test) { t ->
      t.afterSuite { descriptor, result ->
        def indicator = "\u001B[32m✓\u001b[0m"
        if (result.failedTestCount) {
          indicator = "\u001B[31m✘\u001b[0m"
        }
        if (descriptor.className) {
          logger.lifecycle("$indicator Test ${descriptor.name}; Executed: ${result.testCount}/\u001B[32m${result.successfulTestCount}\u001B[0m/\u001B[31m${result.failedTestCount}\u001B[0m")
        }
      }
    }
  }

  private def configureRepositories(Project project) {
    if (!project.hasProperty('offline')) {
      // public first so most artifacts resolve right away (to speed up the builds)
      project.repositories {
        if (project.hasProperty("nexusBaseUrl")) {
          maven {
            url "${project.property("nexusBaseUrl")}/groups/public"
          }

          // then resolve from alphas, and the less common artifacts
          ["alphas", "releases", "thirdparty", "hidden-thirdparty"].each { r ->
            maven {
              credentials {
                username project.property("nexusUserName")
                password project.property("nexusPassword")
              }
              url "${project.property("nexusBaseUrl")}/repositories/${r}"
            }
          }
        }

        if (project.hasProperty("localMavenRepository")) {
          maven {
            url "${project.getRootDir().toURI().toURL()}mavenrepo/snapshots"
          }
          maven {
            url "${project.getRootDir().toURI().toURL()}mavenrepo/releases"
          }
        }

        // Last search local.
        mavenLocal()
      }
    }
  }

  private def configureJava(Project project) {
    configureLanguageLevel(project)
    project.configurations.compile.transitive = false

    project.tasks.withType(JavaCompile).each { t ->
      t.options.encoding = "UTF-8"
      t.options.incremental = true
    }

  }

  private void configureLanguageLevel(Project project) {
    if (!project.hasProperty("languageLevel")) {
      throw new GradleException("Could not determine the Java language level, please add `languageLevel=(1.8|11)` to ${project.rootProject.projectDir}/gradle.properties")
    }
    def languageLevel = project.property("languageLevel")

    project.sourceCompatibility = languageLevel
    project.targetCompatibility = languageLevel

    def checkJavaTask = project.rootProject.tasks.getByName("checkJavaVersion")
    project.tasks.withType(JavaCompile)*.dependsOn(checkJavaTask)
  }

  private def configureScala(Project project) {
    project.configurations.compile.transitive = true

    // making scala responsible for compile scala and java
    project.sourceSets.main.scala.srcDir "src/main/java"
    project.sourceSets.main.java.srcDirs = []

    project.sourceSets.test.scala.srcDir "src/test/java"
    project.sourceSets.test.java.srcDirs = []

    // Disable Java Compilation
    project.tasks.withType(JavaCompile)*.onlyIf { false }

    project.tasks.withType(ScalaCompile) { task ->
      task.scalaCompileOptions.additionalParameters = ["-feature"]
      task.scalaCompileOptions.forkOptions.memoryMaximumSize = '1g'
    }
  }
}
