package ai.digital.deploy.metrics.rest

import ai.digital.deploy.metrics.api.DeploymentMetricsService
import ai.digital.deploy.metrics.repository.DeploymentMetricsRepository
import ai.digital.deploy.metrics.model.{DeploymentMetrics, DeploymentMetricsFilter, DeploymentPluginMetrics}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller
import com.xebialabs.deployit.engine.api.dto.Ordering
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN

import scala.collection.mutable.ListBuffer

@Controller
class DeploymentMetricsServiceResource(@Autowired deploymentMetricsRepository: DeploymentMetricsRepository)
                                      extends AbstractSecuredResource with DeploymentMetricsService{
  override def listDeployments(filter: DeploymentMetricsFilter, paging: Paging, order: Ordering):
                                                                      List[DeploymentMetrics] = {
    checkPermission(ADMIN)
    if (filter.plugins != null && filter.plugins.size > 0) {
      val deploymentMetrics = deploymentMetricsRepository.findDeployments(filter.path, filter.plugins, Some(filter.owner),
        Some(filter.startDate), Some(filter.taskType), paging, order)
      val deploymentPluginMetrics = deploymentMetricsRepository.getPluginsData(filter.path, filter.plugins, Some(filter.owner),
        Some(filter.startDate), Some(filter.taskType), paging, order)
      combine(deploymentMetrics, deploymentPluginMetrics)
    } else {
      deploymentMetricsRepository.findDeployments(filter.path, filter.plugins, Some(filter.owner),
                        Some(filter.startDate), Some(filter.taskType), paging, order)
    }
  }

  override def getDeploymentsCount(): Int = {
    checkPermission(ADMIN)
    deploymentMetricsRepository.getDeploymentCount()
  }

  override def getUsersCountByDeployment(username: String): Int = {
    checkPermission(ADMIN)
    deploymentMetricsRepository.getUsersCount(username)
  }

  override def getPluginsData(filter: DeploymentMetricsFilter,
                                      paging: Paging, order: Ordering): List[DeploymentPluginMetrics] = {
    checkPermission(ADMIN)
    deploymentMetricsRepository.getPluginsData(filter.path, filter.plugins, Some(filter.owner),
      Some(filter.startDate), Some(filter.taskType), paging, order)
  }

  override def getPluginsCountByDeployment(): Int = {
    checkPermission(ADMIN)
    deploymentMetricsRepository.getPluginsCount()
  }

  private def combine(deploymentMetrics: List[DeploymentMetrics],
                      deploymentPluginMetrics: List[DeploymentPluginMetrics]) : List[DeploymentMetrics] = {
    val acc = new ListBuffer[DeploymentMetrics]
    deploymentMetrics.foreach(deploymentMetrics => {
      val plugins = deploymentPluginMetrics.filter(deploymentPluginMetrics => deploymentPluginMetrics.taskId == deploymentMetrics.taskId)
                                                  .collect(deploymentPluginMetrics => deploymentPluginMetrics.ciType)
      acc += DeploymentMetrics(deploymentMetrics.taskId, deploymentMetrics.ciType,
        deploymentMetrics.path, Some(plugins).getOrElse(List.empty),
        deploymentMetrics.owner, deploymentMetrics.taskType, deploymentMetrics.startDate, deploymentMetrics.status)
    })
    acc.toList
  }
}
