package ai.digital.deploy.metrics.rest

import ai.digital.deploy.metrics.api.CiMetricsService
import ai.digital.deploy.metrics.config.Constants
import ai.digital.deploy.metrics.model.{CiCount, CiMetrics, CiMetricsFilter, CiPluginMetrics, FoldersCount, PluginMetrics, PluginsCountByCi, UsersCountByCi}
import ai.digital.deploy.metrics.repository.CiMetricsRepository
import com.xebialabs.deployit.core.rest.api.MetadataServiceImpl
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource
import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.security.permission.PlatformPermissions.{ADMIN}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

@Controller
class CiMetricsServiceResource(@Autowired ciMetricsRepository: CiMetricsRepository) extends AbstractSecuredResource with CiMetricsService {
  override def listConfigurationItems(filter: CiMetricsFilter, paging: Paging, order: dto.Ordering): List[CiPluginMetrics] = {
    checkPermission(ADMIN)
    Option(filter.path) match {
      case Some(filter.path) if filter.path.nonEmpty =>
        val ciTypes = findCiTypes(filter.path)
        val ciMetrics = ciMetricsRepository.findCis(ciTypes, Some(filter.path), Some(filter.createdBy), Some(filter.createdAt), paging, order)
        val ciIds = ciMetrics.collect(metric => metric.id)
        val fragmentedCiIds = ciIds.grouped(1000).toList
        val pluginMetrics = listPlugins(filter.path,fragmentedCiIds, Some(filter.plugins))
        combine(ciMetrics, pluginMetrics)
      case _ => throw new IllegalArgumentException("Path cannot be empty")
    }
  }

  override def getConfigurationItemsCount(filter: CiMetricsFilter): CiCount = {
    checkPermission(ADMIN)
    Option(filter.path) match {
      case Some(filter.path) if filter.path.nonEmpty =>
        val ciTypes = findCiTypes(filter.path)
        ciMetricsRepository.getCisCount(ciTypes, Some(filter.path), Some(filter.createdBy), Some(filter.createdAt), Some(filter.plugins))
      case _ => throw new IllegalArgumentException("Path cannot be empty")
    }
  }

  override def getUsersCountByCI: UsersCountByCi = {
    checkPermission(ADMIN)
    ciMetricsRepository.getUsersCount(getAllCiTypes)
  }

  override def getSubFoldersCountByCI(path: String): FoldersCount = {
    checkPermission(ADMIN)
    Option(path) match {
      case Some(path) if path.nonEmpty => ciMetricsRepository.getSubFoldersCount(Some(path))
      case _ => throw new IllegalArgumentException("Parameter Path cannot be empty")
    }
  }

  private def findInfrastructureTypes: List[String] = {
    val metadataServiceImpl = new MetadataServiceImpl
    val descriptors = metadataServiceImpl.listDescriptors()
    descriptors.asScala.collect { case descriptor if (descriptor.getRootName == Constants.Infrastructure) => descriptor.getType.toString }.toList
  }

  private def findCiTypes(path: String): List[String] = {
    if (path.startsWith(Constants.ApplicationRoot)) List(Constants.ApplicationType)
    else if (path.startsWith(Constants.EnvironmentRoot)) List(Constants.EnvironmentType)
    else if (path.startsWith(Constants.InfrastructureRoot)) findInfrastructureTypes
    else throw new IllegalArgumentException("Path doesn't match any of the CI Paths")
  }

  private def listPlugins(path: String, fragmentedCiIds: List[List[Int]], plugins: Option[List[String]]): List[PluginMetrics] = {
    var acc = new ListBuffer[PluginMetrics]
    if (path.startsWith(Constants.ApplicationRoot)) {
      fragmentedCiIds.foreach(ciIds => acc = acc ++ ciMetricsRepository.getPluginsDataByApplication(ciIds, plugins))
    } else if (path.startsWith(Constants.EnvironmentRoot)) {
      fragmentedCiIds.foreach(ciIds => acc = acc ++ ciMetricsRepository.getPluginsDataByEnvironment(ciIds, plugins))
    } else if (path.startsWith(Constants.InfrastructureRoot)) {
      fragmentedCiIds.foreach(ciIds => acc = acc ++ ciMetricsRepository.getPluginsDataByInfra(ciIds, plugins))
    }
    acc.toList
  }

  private def getAllCiTypes: List[String] = List(Constants.ApplicationType, Constants.EnvironmentType) ++ findInfrastructureTypes

  private def combine(ciMetrics: List[CiMetrics], pluginMetrics: List[PluginMetrics]) : List[CiPluginMetrics] = {
    var acc = new ListBuffer[CiPluginMetrics]
    ciMetrics.foreach(ciMetric => {
      val plugins = pluginMetrics.filter(pluginMetric => pluginMetric.ciId == ciMetric.id).collect(pluginMetric => pluginMetric.pluginType)
      acc = acc :+ CiPluginMetrics(ciMetric.id, ciMetric.name, ciMetric.ciType, ciMetric.path, Some(plugins).getOrElse(List.empty), ciMetric.createdBy,
        ciMetric.createdAt, ciMetric.modifiedBy,  ciMetric.modifiedAt)
    })
  acc.toList
  }

  override def getPluginsCountByCI: PluginsCountByCi = {
    checkPermission(ADMIN)
    PluginsCountByCi(ciMetricsRepository.getPluginsCountByApplication + ciMetricsRepository.getPluginsCountByEnvironment +
      ciMetricsRepository.getPluginsCountByInfra(findInfrastructureTypes))
  }

}
