package com.xebialabs.deployit.support.application

import java.util
import java.util.stream.Collectors

import com.xebialabs.deployit.plugin.api.udm.{Dictionary, Environment}
import com.xebialabs.deployit.repository.sql.CiRepository
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service


import scala.jdk.CollectionConverters._

@Service
class EnvironmentSupportReportService @Autowired()(environmentDetailReportRepository: EnvironmentSupportReportRepository,
                                                   @Autowired repository: CiRepository
                                                  ) extends Logging {
  def getEnvironmentsWithMembersCount: Environments = {
    val paths = environmentDetailReportRepository.getEnvironmentsPath
    val environments = repository.read[Environment](paths, 1, useCache = true, decryptPasswords = false, skipNotExistingCis = true)
    val dictionaries = getAllEnvironmentsDictionaries(environments)
    Environments(environments.size(), getEnvironmentsConsolidatedMembersCount(environments, dictionaries))
  }

  private def getAllEnvironmentsDictionaries(environments: util.List[Environment]): List[Dictionary] = {
    val dictionariesIds = environments.stream()
      .flatMap(environment => environment.getDictionaries.asScala.map(dic => dic.getId).asJava.stream())
      .distinct()
      .collect(Collectors.toList[String])
    repository.read[Dictionary](dictionariesIds, 1, useCache = true, decryptPasswords = false, skipNotExistingCis = true).asScala.toList
  }

  private def getEnvironmentsConsolidatedMembersCount(environments: util.List[Environment], dictionaries: List[Dictionary]): List[EnvironmentsDetails] = {
    environments.asScala.map {
      environment =>
        val environmentDictionaryIds = environment.getDictionaries
          .stream()
          .map(dictionary => dictionary.getId)
          .collect(Collectors.toList[String])
        val environmentDictionariesDetails = dictionaries.asJava
          .stream()
          .filter(dictionary => environmentDictionaryIds.contains(dictionary.getId))
          .collect(Collectors.toList[Dictionary])
        EnvironmentsDetails(environment.getId, environment.getMembers.size(), environmentDictionariesDetails.size(), getConsolidatedDictionaryCount(environmentDictionariesDetails))
    }.toList
  }

  private def getConsolidatedDictionaryCount(dictionaries: util.List[Dictionary]): Map[String, Int] =
    dictionaries.asScala.groupMapReduce {
      case dictionary: Dictionary
        if (dictionary.getRestrictToApplications.size() == 0
          && dictionary.getRestrictToContainers.size() == 0) => "numberOfNonRestricted"
      case dictionary: Dictionary
        if (dictionary.getRestrictToApplications.size() > 0
          && dictionary.getRestrictToContainers.size() == 0) => "numberOfRestrictedApplication"
      case dictionary: Dictionary
        if (dictionary.getRestrictToApplications.size() == 0
          && dictionary.getRestrictToContainers.size() > 0) => "numberOfRestrictedContainer"
      case dictionary: Dictionary
        if (dictionary.getRestrictToApplications.size() > 0
          && dictionary.getRestrictToContainers.size() > 0) => "numberOfBothRestricted"
    }(x => 1)(_ + _)
}

case class EnvironmentsDetails(id: String, numberOfContainers: Int, numberOfDictionaries: Int, dictionaries: Map[String, Int])

case class Environments(numberOfEnvironments: Int, environments: List[EnvironmentsDetails])
