package com.xebialabs.deployit.support.report

import com.xebialabs.deployit.core.service.GeneralSettingsService
import com.xebialabs.deployit.repository.{RepositoryDatabaseMetaDataService}
import com.xebialabs.deployit.{ReleaseInfo, ServerConfiguration}
import com.xebialabs.xlplatform.cluster.NodeState
import com.xebialabs.xlplatform.support.report.ReportDataProvider
import grizzled.slf4j.Logging
import org.joda.time.Duration
import org.joda.time.format.PeriodFormatterBuilder
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils

import java.lang.management.ManagementFactory
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._
import scala.util.Try

@Component
class GeneralDataProvider @Autowired()(val databaseMetaDataService: RepositoryDatabaseMetaDataService,
                                       val dataStatisticsRepository: DataStatisticsRepositoryService,
                                       val serverConfiguration: ServerConfiguration,
                                       val generalSettingsService: GeneralSettingsService)
                                       extends ReportDataProvider with Logging {

  val name: String = "general"

  override def collectStatistics:  Map[String, Any] = {
    val statistics = new GeneralUsageInfo()
    val releaseInfo = ReleaseInfo.getReleaseInfo
    statistics.setDeployVersion(s"${releaseInfo.getVersion} (built at ${releaseInfo.getDate})")
    val generalSettings = generalSettingsService.getGeneralSettings;
    statistics.setInstanceName(if (StringUtils.hasText(generalSettings.instanceName)) generalSettings.instanceName else "Not defined")
    statistics.setServerUrl(serverConfiguration.getServerUrl)
    statistics.setNoHitDBQueryDuration(dataStatisticsRepository.timeDbQueryNoHit)
    statistics.setHitDBQueryDuration(dataStatisticsRepository.timeDbQueryHit)
    statistics.setDatabaseImplementation(databaseMetaDataService.findMainDbImplementation())
    statistics.setReportingDatabaseImplementation(databaseMetaDataService.findReportDbImplementation())

    val osMbean = ManagementFactory.getOperatingSystemMXBean
    val runTimeBean = ManagementFactory.getRuntimeMXBean
    statistics.setVmArguments(runTimeBean.getInputArguments.asScala.mkString(","))
    statistics.setVmName(s"${runTimeBean.getVmName} ${runTimeBean.getVmVendor} ${runTimeBean.getVmVersion} (${runTimeBean.getSpecVersion})")
    statistics.setSystemName(runTimeBean.getName)
    statistics.setUpTime(formatTimeInMS(runTimeBean.getUptime))
    statistics.setOsName(osMbean.getName)
    statistics.setOsArchitecture(osMbean.getArch)
    statistics.setOsAvailableProcessors(osMbean.getAvailableProcessors)
    val runtime = Runtime.getRuntime
    Try(System.gc()).recover { case exception => logger.warn("Unable to GC due to error", exception) }
    statistics.setUsedMemory(runtime.totalMemory() - runtime.freeMemory())
    statistics.setMaxMemory(runtime.maxMemory())
    statistics.setIsNodeActive(NodeState.isActive)
    Map[String, Any]("general" -> statistics)
  }

  def formatTimeInMS(upTimeMS: Long): String = {
    val periodFormatter = new PeriodFormatterBuilder()
      .appendDays()
      .appendSuffix("d")
      .appendHours()
      .appendSuffix("h")
      .appendMinutes()
      .appendSuffix("m")
      .appendSeconds()
      .appendSuffix("s")
      .toFormatter()
    periodFormatter.print(new Duration(upTimeMS).toPeriod())
  }
}

class GeneralUsageInfo {
  @BeanProperty
  var deployVersion: String = _

  @BeanProperty
  var instanceName: String = _

  @BeanProperty
  var serverUrl: String = _

  @BeanProperty
  var vmArguments: String = _

  @BeanProperty
  var vmName: String = _

  @BeanProperty
  var systemName: String = _

  @BeanProperty
  var upTime: String = _

  @BeanProperty
  var osName: String = _

  @BeanProperty
  var osArchitecture: String = _

  @BeanProperty
  var osAvailableProcessors: Int = 0

  @BeanProperty
  var maxMemory: Long = 0

  @BeanProperty
  var usedMemory: Long = 0

  @BeanProperty
  var databaseImplementation: String = _

  @BeanProperty
  var reportingDatabaseImplementation: String = _

  @BeanProperty
  var noHitDBQueryDuration: Long = 0

  @BeanProperty
  var hitDBQueryDuration: Long = 0

  @BeanProperty
  var isNodeActive: Boolean = false
}
