package com.xebialabs.deployit.support.report

import ai.digital.deploy.metrics.repository.CiMetricsRepository
import ai.digital.deploy.sql.model.Report
import com.xebialabs.deployit.core.api.resteasy.Date
import com.xebialabs.xlplatform.support.report.ReportDataProvider
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import com.xebialabs.deployit.core.api.TaskMonitorService
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState
import com.xebialabs.deployit.repository.placeholders.PlaceholderRepository
import com.xebialabs.deployit.security.service.UserStatsService

import scala.beans.BeanProperty
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Component
class DataStatisticsProvider @Autowired () (val dataStatisticsRepository: DataStatisticsRepositoryService,
                                            val ciMetricsRepository: CiMetricsRepository,
                                            val placeholderRepository: PlaceholderRepository,
                                            val taskMonitorService: TaskMonitorService,
                                            val userStatsService: UserStatsService,
                                            val pluginStatisticsRepository: PluginStatisticsRepository)
                            extends ReportDataProvider {
  val name: String = "data"

  override def collectStatistics: Map[String, Any] = {
    Map[String, Any]("data" -> mutable.Map[String, Any](
      "ci" -> collectCiStatistics,
      "deployments" -> collectDeploymentStatistics,
      "tasks" -> collectControlTaskUsageStatistics,
      "folders" -> collectFolderStatistics,
      "users" -> collectUsersAndRoleStatistics,
      "placeholders" -> collectPlaceholderStatistics,
      "plugins" -> collectPluginsStatistics
    ).asJava)
  }

  def collectCiStatistics: CiStatistics = {
    val statistics = new CiStatistics
    val applicationCount = ciMetricsRepository.getCisCount(List("udm.Application"),
      Some("/Applications"), null, null, null)
    statistics.totalApplication = applicationCount.ciCount.toInt
    statistics.totalHosts = dataStatisticsRepository.hostCount
    statistics.totalSatellites = dataStatisticsRepository.satelliteCount
    statistics.totalWorker = dataStatisticsRepository.workersCount
    statistics
  }

  def collectDeploymentStatistics: DeploymentStatistics = {
    val statistics = new DeploymentStatistics
    val fromDate = new Date(DateTime.now().minusDays(30).toDate.toString)
    val toDate = new Date(DateTime.now().toDate.toString)

    // Total Deployment for 24hrs
    val from24hrs = new Date(DateTime.now().minusHours(24).toDate.toString)
    val reportFor24hrs = dataStatisticsRepository.getDeploymentReportForAPeriod(from24hrs, toDate)
    statistics.successfulDeploymentsIn24hrs = getDeploymentCount(reportFor24hrs, "successful")
    statistics.rollbackDeploymentsIn24hrs = getDeploymentCount(reportFor24hrs, "rollback")
    statistics.abortedDeploymentsIn24hrs = getDeploymentCount(reportFor24hrs, "aborted")
    statistics.retriedDeploymentsIn24hrs =  getDeploymentCount(reportFor24hrs, "failed")
    statistics.totalDeploymentFor24hrs = getDeploymentCount(reportFor24hrs, "totalDeployment")

    // Total Deployment for 30 days
    val reportFor30days = dataStatisticsRepository.getDeploymentReportForAPeriod(fromDate, toDate)
    statistics.totalDeploymentFor30days =  getDeploymentCount(reportFor30days, "totalDeployment")
    statistics.averageDeploymentPerDayPerMonth = report.avg(statistics.totalDeploymentFor30days, 30)

    // Total Deployment over a year
    val forAYear = new Date(DateTime.now().minusDays(365).toDate.toString)
    val reportForAYear = dataStatisticsRepository.getDeploymentReportForAPeriod(forAYear, toDate)
    val totalDeploymentForYr = getDeploymentCount(reportForAYear, "totalDeployment")
    statistics.averageDeploymentPerDayPerYr = report.avg(totalDeploymentForYr, 365)
    statistics.totalDeploymentsInLastYear = totalDeploymentForYr
    statistics.successfulDeploymentsInLastYear = getDeploymentCount(reportForAYear, "successful")
    statistics.rollbackDeploymentsInLastYear = getDeploymentCount(reportForAYear, "rollback")
    statistics.abortedDeploymentsInLastYear = getDeploymentCount(reportForAYear, "aborted")
    statistics.retriedDeploymentsInLastYear = getDeploymentCount(reportForAYear, "failed")


    // Pending deployment
    statistics.totalPendingDeployment = dataStatisticsRepository.pendingTaskCount("deployment",
      true, true, true, true)
    statistics.totalScheduledDeployment =  dataStatisticsRepository.pendingTaskCount("deployment",
      true, false, false, false)
    statistics.totalQueuedDeployment = dataStatisticsRepository.pendingTaskCount("deployment",
      false, false, true, false)
    statistics.activeDeployments = dataStatisticsRepository.activeTaskCount("deployment")
    statistics
  }

  def getDeploymentCount(report: Report, deploymentType: String): Int = {
    val successful: Int = report.getLines.get(0).getValues.get("noOfSuccessfulDeployments").toString.toInt
    val rollBack: Int = report.getLines.get(0).getValues.get("noOfRollbacks").toString.toInt
    val failed: Int = report.getLines.get(0).getValues.get("noOfFailedDeployments").toString.toInt
    val aborted: Int = report.getLines.get(0).getValues.get("noOfAbortedDeployments").toString.toInt
    val totalDeployment: Int = successful + rollBack + failed + aborted
    deploymentType match {
      case "successful" => successful
      case "rollback" => rollBack
      case "failed" => failed
      case "aborted" => aborted
      case "totalDeployment" => totalDeployment
    }
  }

  def collectFolderStatistics: FolderStatistics = {
    val statistics = new FolderStatistics
    statistics.applicationFolderCount = ciMetricsRepository.getSubFoldersCount(Some("/Applications")).foldersCount.toInt
    statistics.environmentFolderCount = ciMetricsRepository.getSubFoldersCount(Some("/Environments")).foldersCount.toInt
    statistics.infraFolderCount = ciMetricsRepository.getSubFoldersCount(Some("/Infrastructure")).foldersCount.toInt
    statistics
  }

  def collectControlTaskUsageStatistics: ControlTaskStatistics = {
    val controlTaskStatistics: ControlTaskStatistics = new ControlTaskStatistics

    val totalTasksFor24Hours = dataStatisticsRepository.archivedControlTaskCount(null ,
      DateTime.now().minusDays(1), DateTime.now())
    controlTaskStatistics.setTotalTaskFor24hrs(totalTasksFor24Hours)

    val totalTasksFor30Days = dataStatisticsRepository.archivedControlTaskCount(null ,
      DateTime.now().minusDays(30), DateTime.now())
    controlTaskStatistics.setTotalTaskFor30days(totalTasksFor30Days)

    controlTaskStatistics.setAverageTaskPerDayPerMonth(report.avg(totalTasksFor30Days, 30))

    val totalTasksPerYear = dataStatisticsRepository.archivedControlTaskCount("ALL" ,
      DateTime.now().minusDays(365), DateTime.now())
    controlTaskStatistics.setAverageTaskPerDayPerYr(report.avg(totalTasksPerYear, 365))


    val totalActiveTasks = dataStatisticsRepository.activeTaskCount("Control task")
    controlTaskStatistics.setTotalActiveTasks(totalActiveTasks)

    val pendingCount = dataStatisticsRepository.pendingTaskCount("Control task",
      true, true, true, true)
    controlTaskStatistics.setTotalPendingTasks(pendingCount)

    val scheduledCount = dataStatisticsRepository.pendingTaskCount("Control task",
      true, false, false, false)
    controlTaskStatistics.setTotalScheduledTasks(scheduledCount)

    val queuedCount = dataStatisticsRepository.pendingTaskCount("Control task",
      false, false, true, false)
    controlTaskStatistics.setTotalQueuedTasks(queuedCount)

    val completedCountFor24Hours = dataStatisticsRepository.archivedControlTaskCount(TaskExecutionState.DONE.name(),
      DateTime.now().minusDays(1), DateTime.now())
    controlTaskStatistics.setTotalSuccessfulTasksIn24Hours(completedCountFor24Hours)

    val abortedCountFor24Hours = dataStatisticsRepository.archivedControlTaskCount(TaskExecutionState.CANCELLED.name(),
      DateTime.now().minusDays(1), DateTime.now())
    controlTaskStatistics.setTotalAbortedTasksIn24Hours(abortedCountFor24Hours)

    controlTaskStatistics
  }

  def collectUsersAndRoleStatistics: UserAndRoleStatistics = {
    val statistics  = new UserAndRoleStatistics
    statistics.totalGlobalRoles = dataStatisticsRepository.totalGlobalRoleCount

    val userStatsByProfile = userStatsService.prepareUserStatsByUserProfile
    statistics.numberOfUsers = userStatsByProfile("allUsers")
    statistics.numberOfLicensedUsers = userStatsByProfile("usersEnabled")
    statistics.numberOfLicensedExternalUsers = userStatsByProfile("usersEnabledWithExternal")
    statistics.numberOfLicensedInternalUsers = userStatsByProfile("usersEnabledWithInternal")
    statistics
  }

  def collectPlaceholderStatistics: PlaceholderStatistics = {
    val statistics  = new PlaceholderStatistics
    statistics.placeholdersCount = dataStatisticsRepository.getAllPlaceholdersCount
    statistics.avgPlaceholdersCountByDeployedApp = dataStatisticsRepository.getAvgPlaceholdersCount
    statistics.appWithMaxPlaceholders = dataStatisticsRepository.getApplicationsWithMaxPlaceholders(5)
                                                .map{ case (k, v) => k + " - " + v.toString }.mkString("\n")
    statistics
  }

  def collectPluginsStatistics: PluginStatistics = {
    val statistics = new PluginStatistics
    statistics.pluginsDataByDeployment = pluginStatisticsRepository.getPluginDataByDeploymentByDuration(
      DateTime.now().minusDays(180), DateTime.now())
    statistics
  }
}

class CiStatistics {
  @BeanProperty
  var totalApplication: Int = 0

  @BeanProperty
  var totalHosts: Int = 0

  @BeanProperty
  var totalSatellites: Int = 0

  @BeanProperty
  var totalWorker: Int = 0
}

class FolderStatistics {
  @BeanProperty
  var applicationFolderCount: Int = 0

  @BeanProperty
  var environmentFolderCount: Int = 0

  @BeanProperty
  var infraFolderCount: Int = 0
}

class DeploymentStatistics {
  @BeanProperty
  var totalDeploymentFor24hrs: Int = 0
  @BeanProperty
  var totalDeploymentFor30days: Int = 0
  @BeanProperty
  var averageDeploymentPerDayPerYr: Float = 0
  @BeanProperty
  var averageDeploymentPerDayPerMonth: Float = 0
  @BeanProperty
  var totalPendingDeployment: Int = 0
  @BeanProperty
  var totalScheduledDeployment: Int = 0
  @BeanProperty
  var totalQueuedDeployment: Int = 0
  @BeanProperty
  var activeDeployments: Int = 0
  @BeanProperty
  var successfulDeploymentsIn24hrs: Int = 0
  @BeanProperty
  var retriedDeploymentsIn24hrs: Int = 0
  @BeanProperty
  var rollbackDeploymentsIn24hrs: Int = 0
  @BeanProperty
  var abortedDeploymentsIn24hrs: Int = 0
  @BeanProperty
  var totalDeploymentsInLastYear: Int = 0
  @BeanProperty
  var successfulDeploymentsInLastYear: Int = 0
  @BeanProperty
  var retriedDeploymentsInLastYear: Int = 0
  @BeanProperty
  var rollbackDeploymentsInLastYear: Int = 0
  @BeanProperty
  var abortedDeploymentsInLastYear: Int = 0
}
class ControlTaskStatistics {
  @BeanProperty
  var totalTaskFor24hrs: Int = 0
  @BeanProperty
  var totalTaskFor30days: Int = 0
  @BeanProperty
  var averageTaskPerDayPerYr: Float = 0
  @BeanProperty
  var averageTaskPerDayPerMonth: Float = 0
  @BeanProperty
  var totalActiveTasks: Int = 0
  @BeanProperty
  var totalPendingTasks: Int = 0
  @BeanProperty
  var totalScheduledTasks: Int = 0
  @BeanProperty
  var totalQueuedTasks: Int = 0
  @BeanProperty
  var totalSuccessfulTasksIn24Hours: Int = 0
  @BeanProperty
  var totalAbortedTasksIn24Hours: Int = 0
}
class UserAndRoleStatistics {
  @BeanProperty
  var totalGlobalRoles: Int = 0
  @BeanProperty
  var numberOfUsers: Int = 0
  @BeanProperty
  var numberOfLicensedUsers: Int = 0
  @BeanProperty
  var numberOfLicensedInternalUsers: Int = 0
  @BeanProperty
  var numberOfLicensedExternalUsers: Int = 0


}

class PlaceholderStatistics {
  @BeanProperty
  var placeholdersCount: Int = 0
  @BeanProperty
  var appWithMaxPlaceholders: String = ""
  @BeanProperty
  var avgPlaceholdersCountByDeployedApp: Int = 0
}

class PluginStatistics {
  @BeanProperty
  var pluginsDataByDeployment: String = ""
}
