package com.xebialabs.deployit.support.report.sql

import ai.digital.deploy.sql.model.Report
import com.google.common.base.Strings
import com.xebialabs.deployit.core.api.InternalReportProxy
import com.xebialabs.deployit.core.api.resteasy.Date
import com.xebialabs.deployit.core.sql.spring.{DeployJdbcTemplate, Setter}
import com.xebialabs.deployit.core.sql.{ColumnName, Queries, SchemaInfo, SelectBuilder, SelectFragmentBuilder, Selectable, SqlFunction, SqlLiteral, TableName, SqlCondition => cond}
import com.xebialabs.deployit.engine.tasker.repository.sql.{PendingTaskSchema, WorkersSchema}
import com.xebialabs.deployit.repository.sql.persisters.{HostSchema, SatelliteSchema}
import com.xebialabs.deployit.repository.sql.placeholders.{PlaceholderSchema, PlaceholdersSchema}
import com.xebialabs.deployit.security.{Role, RoleService}
import com.xebialabs.deployit.sql.base.schema.{ActiveTaskSchema, CIS}
import com.xebialabs.deployit.support.report.DataStatisticsRepositoryService
import com.xebialabs.deployit.task.archive.sql.schema.ArchivedControlTasks
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import java.util
import javax.sql.DataSource
import scala.jdk.CollectionConverters.CollectionHasAsScala

@Repository
class DataStatisticsRepository @Autowired() (@Autowired
                              @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                              @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo,
                              @Qualifier("reportingDataSource") val dataSource: DataSource,
                              @Qualifier("reportingSchema") val reportingSchemaInfo: SchemaInfo,
                              val internalReportProxy: InternalReportProxy,
                              val roleService: RoleService)
  extends DataStatisticsRepositoryService with Queries {

  val TEST_QUERY = sqlb"${DataSchema.component} FROM ${DataSchema.tableName}"
  val HOST_COUNT = sqlb"select count(*) from  ${HostSchema.tableName}"
  val SATELLITES_COUNT = sqlb"select count(*) from  ${SatelliteSchema.tableName}"
  val WORKERS_COUNT = sqlb"select count(*) from  ${WorkersSchema.tableName}"
  lazy val SELECT_PLACEHOLDERS_COUNT : String = sqlb"select count(*) from ${PlaceholderSchema.tableName}, ${CIS.tableName} where ${CIS.path} = ${PlaceholderSchema.ciPath}"
  lazy val SELECT_PLACEHOLDERS_AVG_BY_APP : String = sqlb"select avg(maxcount) from (select count(distinct ${PlaceholdersSchema.key}) " +
    sqlb"as maxcount from ${PlaceholdersSchema.tableName} group by ${PlaceholdersSchema.deployedAppName}) a"
  lazy val SELECT_APP_WITH_MAX_PLACEHOLDERS : String = sqlb"${PlaceholdersSchema.deployedAppName}, count(distinct ${PlaceholdersSchema.key}) " +
    sqlb"as maxcount from ${PlaceholdersSchema.tableName} group by ${PlaceholdersSchema.deployedAppName} order by maxcount desc"
  lazy val reportingTemplate = new DeployJdbcTemplate(dataSource, false)

  def timeDbQueryNoHit: Long = {
    var sql = ""
    schemaInfo.sqlDialect.toString match {
      case "oracle" => sql = "SELECT CURRENT_TIMESTAMP FROM dual"
      case "db2" => sql = "SELECT CURRENT_TIMESTAMP FROM sysibm.sysdummy1"
      case "derby" => sql = "VALUES CURRENT_TIMESTAMP"
      case _ => sql = "SELECT CURRENT_TIMESTAMP"
    }
    val startTime = System.currentTimeMillis()
    jdbcTemplate.queryForObject(sql, classOf[String])
    val endTime = System.currentTimeMillis()
    endTime - startTime
  }

  def timeDbQueryHit: Long = {
    val builder = new SelectFragmentBuilder(TEST_QUERY)
    val startTime = System.currentTimeMillis()
    jdbcTemplate.queryForList(builder.query)
    val endTime = System.currentTimeMillis()
    endTime - startTime
  }

  def hostCount: Int = jdbcTemplate.queryForObject(HOST_COUNT, classOf[Number]).intValue()
  def satelliteCount: Int = jdbcTemplate.queryForObject(SATELLITES_COUNT, classOf[Number]).intValue()
  def workersCount: Int = jdbcTemplate.queryForObject(WORKERS_COUNT, classOf[Number]).intValue()

  override def getDeploymentReportForAPeriod(fromDate: Date, toDate: Date): Report = {
    val report : Report = internalReportProxy.getDashboardWidgetReport(
      "deploymentSuccess", fromDate, toDate)
    report
  }

  def getActiveDeploymentCount: Int = activeTaskCount("deployment")

  override def totalGlobalRoleCount: Int =  {
    val roleList: util.List[Role] = roleService.getRoles(null, null, null)
    roleList.size()
  }

  override def archivedControlTaskCount(status: String, begin: DateTime, end: DateTime): Int = {
    val selectBuilder = createSelectBuilder(ArchivedControlTasks.tableName)
    addStartDateFilter(selectBuilder, begin, end)
    addCountFilter(selectBuilder)
    Option(status) match {
      case Some(status) if status.nonEmpty =>
        addStringLikeConstraint(selectBuilder, ArchivedControlTasks.status, status)
      case _ =>
    }

    executeReportingCountQuery(selectBuilder)
  }

  override def pendingTaskCount(description: String, includeScheduled: Boolean, isScheduledNull: Boolean, includeQueued: Boolean
                                , isQueuedNull: Boolean): Int = {
    val selectBuilder = createSelectBuilder(PendingTaskSchema.tableName)
    addCountFilter(selectBuilder)
    addStringLikeConstraint(selectBuilder, PendingTaskSchema.task_description, description)
    if (includeScheduled && isScheduledNull)
      addScheduledDateNullFilter(selectBuilder)
    if (includeScheduled && !isScheduledNull)
      addScheduledDateNotNullFilter(selectBuilder)
    if (includeQueued && isQueuedNull)
      addQueuedDateNullFilter(selectBuilder)
    if (includeQueued && !isQueuedNull)
      addQueuedDateNotNullFilter(selectBuilder)

    executeCountQuery(selectBuilder)
  }

  def executeCountQuery(selectBuilder: SelectBuilder): Int = jdbcTemplate.query(selectBuilder.query,
                                                                    Setter(selectBuilder.parameters),
    (rs: ResultSet, rowNum: Int) => {
      rs.getInt(1)
    }).get(0)

  @Transactional(value = "reportingTransactionManager", readOnly = true)
  def executeReportingCountQuery(selectBuilder: SelectBuilder): Int = reportingTemplate.query(selectBuilder.query,
                                                                    Setter(selectBuilder.parameters),
    (rs: ResultSet, rowNum: Int) => {
      rs.getInt(1)
    }).get(0)

   override def activeTaskCount(description: String): Int = {
    val selectBuilder = createSelectBuilder(ActiveTaskSchema.tableName)
    addCountFilter(selectBuilder)
    addStringLikeConstraint(selectBuilder, ArchivedControlTasks.description, description)
    executeCountQuery(selectBuilder)
  }

  def createSelectBuilder(table: TableName): SelectBuilder = new SelectBuilder(table)

  def addStartDateFilter(selectBuilder: SelectBuilder, begin: DateTime, end: DateTime): SelectBuilder = selectBuilder
    .where(cond.between(ArchivedControlTasks.start_date, begin, end))

  def addCountFilter(selectBuilder: SelectBuilder): SelectBuilder = selectBuilder.select(new SqlLiteral("count(*)"))

  def addStringLikeConstraint(selectBuilder: SelectBuilder, field: Selectable, value: String): Unit = {
    if (!Strings.isNullOrEmpty(value))
      selectBuilder.where(cond.like(SqlFunction.lower(field), s"%${value.toLowerCase}%"))
  }

  def addScheduledDateNotNullFilter(selectBuilder: SelectBuilder): SelectBuilder = selectBuilder
    .where(cond.equalsNotNull(PendingTaskSchema.scheduled_date))

  def addScheduledDateNullFilter(selectBuilder: SelectBuilder): SelectBuilder = selectBuilder
    .where(cond.equalsNull(PendingTaskSchema.scheduled_date))

  def addQueuedDateNotNullFilter(selectBuilder: SelectBuilder): SelectBuilder = selectBuilder
    .where(cond.equalsNotNull(PendingTaskSchema.queued_date))

  def addQueuedDateNullFilter(selectBuilder: SelectBuilder): SelectBuilder = selectBuilder
    .where(cond.equalsNull(PendingTaskSchema.queued_date))

  def getApplicationsWithMaxPlaceholders(limit: Int): Map[String,Int] = {
    val selectBuilder = new SelectFragmentBuilder(SELECT_APP_WITH_MAX_PLACEHOLDERS)
      .limit(limit)
    jdbcTemplate.query(
      selectBuilder.query,
      new RowMapper[(String, Int)]() {
        override def mapRow(rs: ResultSet, rowNum: Int): (String, Int) = {
          (rs.getString(1), rs.getInt(2))
        }
      }
    ).asScala.toMap
  }

  def getAllPlaceholdersCount: Int = {
    jdbcTemplate.query(
      SELECT_PLACEHOLDERS_COUNT,
      (rs: ResultSet, _: Int) => rs.getInt(1)).asScala.headOption.getOrElse(0)
  }

  def getAvgPlaceholdersCount: Int = {
    jdbcTemplate.query(
      SELECT_PLACEHOLDERS_AVG_BY_APP,
      (rs: ResultSet, _: Int) => Math.ceil(rs.getDouble(1)).intValue()).get(0)
  }
}

object DataSchema {
  val tableName: TableName = TableName("XL_VERSION")
  val component: ColumnName = ColumnName("component")
  val version: ColumnName = ColumnName("version")
}
