package com.xebialabs.deployit.support.report.sql

import com.xebialabs.deployit.core.sql.spring.DeployJdbcTemplate
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo}
import com.xebialabs.deployit.support.report.PluginStatisticsRepository
import com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{RowMapper, RowMapperResultSetExtractor}
import org.springframework.stereotype.Repository

import java.net.URLDecoder
import java.nio.charset.StandardCharsets
import java.sql.{ResultSet, Timestamp}
import javax.sql.DataSource
import scala.jdk.CollectionConverters.CollectionHasAsScala

@Repository
class SqlPluginStatisticsRepository @Autowired()(@Qualifier("reportingDataSource") val dataSource: DataSource,
                                   @Qualifier("reportingSchema") implicit val schemaInfo: SchemaInfo) extends
  PluginStatisticsRepository with Queries {

  private val reportingTemplate = new DeployJdbcTemplate(dataSource, false)

  override def getPluginDataByDeploymentByDuration(startTime: DateTime, endTime: DateTime): String = {
    val deplPluginData = queryPluginDataByDeploymentByDuration(startTime, endTime)
    deplPluginData.filter(plugin => plugin.pluginName != null && plugin.pluginName.nonEmpty)
      .map(plugin => plugin.copy(decodePluginName(plugin.pluginName)))
      .map(_.productIterator.mkString("|")).mkString(",")
  }

  def decodePluginName(name: String): String = {
      try {
        val url = URLDecoder.decode(name, StandardCharsets.UTF_8.name())
        val lastIndex = url.lastIndexOf("\\")
        if (lastIndex > -1) {
          url.substring(lastIndex + 1)
        } else {
          url
        }
      } catch {
        case _: Exception => name
      }
    }

  def queryPluginDataByDeploymentByDuration(startTime: DateTime, endTime: DateTime): Seq[DeplPluginData] = {
      reportingTemplate.query(SELECT_DEPL_COUNT_BY_PLUGIN, new RowMapperResultSetExtractor(deplPluginMapper),
        new Timestamp(startTime.getMillis), new Timestamp(endTime.getMillis)).asScala.toSeq
  }

  case class DeplPluginData(pluginName: String,
                            pluginVersion: String,
                            pluginType: String,
                            deplCount: Int)

  private val deplPluginMapper: RowMapper[DeplPluginData] = (rs: ResultSet, _: Int) =>
    DeplPluginData(
      pluginName = rs.getString(1),
      pluginVersion = rs.getString(2),
      pluginType = rs.getString(3),
      deplCount = rs.getInt(4))

  lazy val SELECT_DEPL_COUNT_BY_PLUGIN : String =
    sqlb"select b.${ArchivedDeploymentTasks.Plugins.plugin_name}, " +
      sqlb"b.${ArchivedDeploymentTasks.Plugins.plugin_version}, " +
      sqlb"b.${ArchivedDeploymentTasks.Plugins.plugin_type}, " +
      sqlb"count(distinct(b.${ArchivedDeploymentTasks.Plugins.task_id})) as taskcount " +
      sqlb"from ${ArchivedDeploymentTasks.tableName} a JOIN" +
      sqlb"${ArchivedDeploymentTasks.Plugins.tableName} b " +
      sqlb"on a.${ArchivedDeploymentTasks.task_id} = b.${ArchivedDeploymentTasks.Plugins.task_id} " +
      sqlb"where a.${ArchivedDeploymentTasks.start_date} >= ? and a.${ArchivedDeploymentTasks.end_date} <= ? " +
      sqlb"group by b.${ArchivedDeploymentTasks.Plugins.plugin_name}, b.${ArchivedDeploymentTasks.Plugins.plugin_version}, " +
      sqlb"b.${ArchivedDeploymentTasks.Plugins.plugin_type}"
}
