package com.xebialabs.deployit.support.report.sql

import com.xebialabs.deployit.core.sql.{SqlCondition => cond, _}
import com.xebialabs.deployit.core.sql.spring.{DeployJdbcTemplate, Setter}
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication
import com.xebialabs.deployit.repository.sql.base.pathToId
import com.xebialabs.deployit.support.report.{TopNDeployedApplicationData, TopNSupportReportRepository, TopTaskData}
import ai.digital.deploy.tasker.common.TaskType
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper, RowMapperResultSetExtractor}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import javax.sql.DataSource
import scala.jdk.CollectionConverters._

@Repository
@Transactional(value = "reportingTransactionManager")
private class SqlTopNSupportReportRepository @Autowired()(val mainTemplate: JdbcTemplate,
                                                          @Qualifier("mainSchema") schemaInfo: SchemaInfo,
                                                          @Qualifier("reportingDataSource") val dataSource: DataSource,
                                                          @Qualifier("reportingSchema") val reportingSchemaInfo: SchemaInfo)
  extends TopNSupportReportRepository {

  private val topNDeployedApplicationMapper: RowMapper[TopNDeployedApplicationData] = (rs: ResultSet, _: Int) =>
    TopNDeployedApplicationData(
      amountOfDeployeds = rs.getInt(1),
      deployedApplication = rs.getInt(2),
      deployedApplicationId = pathToId(rs.getString(3)),
      environment = rs.getInt(4),
      environmentId = pathToId(rs.getString(5))
    )

  private val topTasksMapper: RowMapper[TopTaskData] = (rs: ResultSet, _: Int) =>
    TopTaskData(
      taskId = rs.getString(1),
      durationMillis = rs.getLong(2)
    )

  private val reportingTemplate = new DeployJdbcTemplate(dataSource, false)

  private def mainSqlSupportReportQuery(begin: DateTime, end: DateTime): MainSqlTopNSupportReportQuery =
    new MainSqlTopNSupportReportQuery(begin, end)(schemaInfo)

  private def reportingSqlSupportReportQuery(begin: DateTime, end: DateTime): ReportingTopNSqlQuery =
    new ReportingTopNSqlQuery(begin, end)(reportingSchemaInfo)

  @Transactional(value = "reportingTransactionManager", readOnly = true)
  override def topNTasks(begin: DateTime, end: DateTime, topN: Int,
                         deployedApplicationCi: DeployedApplication, topApplication: TopNDeployedApplicationData): Seq[TopTaskData] = {
    val reportingQuery = reportingSqlSupportReportQuery(begin, end)
    val builder = reportingQuery.topNTasks(topN, deployedApplicationCi, topApplication)
    reportingTemplate.query(builder.query, Setter(builder.parameters, Setter.setStringForReporting), new RowMapperResultSetExtractor(topTasksMapper, topN))
      .asScala
      .toSeq
  }

  @Transactional(value = "mainTransactionManager", readOnly = true)
  override def topNDeployedApplications(begin: DateTime, end: DateTime, topN: Int): Seq[TopNDeployedApplicationData] = {
    val mainQuery = mainSqlSupportReportQuery(begin, end)
    val builder = mainQuery.topNDeployedApplications(topN)
    mainTemplate.query(builder.query, Setter(builder.parameters), new RowMapperResultSetExtractor(topNDeployedApplicationMapper, topN))
      .asScala
      .toSeq
  }
}

class ReportingTopNSqlQuery(val begin: DateTime, val end: DateTime)(implicit val schemaInfo: SchemaInfo) {

  import com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks._

  private def notRetried(selectBuilder: SelectBuilder) = selectBuilder.where(cond.equals(failure_count, 0))

  private def deployTaskTypesOtherThanRollback(selectBuilder: SelectBuilder) =
    selectBuilder.where(cond.in(task_type, List(TaskType.INITIAL, TaskType.UPGRADE, TaskType.UNDEPLOY)))

  def topNTasks(topN: Int, deployedApplicationCi: DeployedApplication, topApplication: TopNDeployedApplicationData): AbstractQueryBuilder = {
    val selectBuilder = new SelectBuilder(tableName)
      .select(task_id).select(duration)
      .where(cond.equals(environment_internal_id, topApplication.environment))
      .where(cond.equals(main_application, deployedApplicationCi.getName))
      .where(cond.between(end_date, begin, end))
      .orderBy(OrderBy.desc(duration))
      .showPage(1, topN)

    notRetried(deployTaskTypesOtherThanRollback(selectBuilder))
  }
}

class MainSqlTopNSupportReportQuery(val begin: DateTime, val end: DateTime)(implicit val schemaInfo: SchemaInfo) {

  import com.xebialabs.deployit.core.sql.{SqlCondition => cond, SqlFunction => func, _}
  import com.xebialabs.deployit.sql.base.schema.{CIS, CI_PROPERTIES}

  def topNDeployedApplications(topN: Int): AbstractQueryBuilder = {

    val deployedAppAlias = "deployed_app"
    val deployedAppPropsAlias = "deployed_app_props"
    val environmentAlias = "environment"

    val selectBuilder = new SelectBuilder(CIS.tableName).as(deployedAppAlias)
      .select(func.countAll)
      .select(CIS.ID.tableAlias(deployedAppAlias))
      .select(CIS.path.tableAlias(deployedAppAlias))
      .select(CIS.parent_id.tableAlias(deployedAppAlias))
      .select(CIS.path.tableAlias(environmentAlias).columnAlias("env_path"))
      .where(cond.between(CIS.modified_at.tableAlias(deployedAppAlias), begin, end))
      .where(cond.equals(CI_PROPERTIES.name.tableAlias(deployedAppPropsAlias), "deployeds"))
      .orderBy(OrderBy.desc(func.countAll))
      .groupBy(CIS.ID.tableAlias(deployedAppAlias))
      .groupBy(CIS.path.tableAlias(deployedAppAlias))
      .groupBy(CIS.parent_id.tableAlias(deployedAppAlias))
      .groupBy(CIS.path.tableAlias(environmentAlias))

    new JoinBuilder(selectBuilder)
      .join(
        new SelectBuilder(CI_PROPERTIES.tableName).as(deployedAppPropsAlias),
        cond.equals(CIS.ID.tableAlias(deployedAppAlias), CI_PROPERTIES.ci_id.tableAlias(deployedAppPropsAlias)),
        JoinType.Left
      )
      .join(
        new SelectBuilder(CIS.tableName).as(environmentAlias),
        cond.equals(CIS.parent_id.tableAlias(deployedAppAlias), CIS.ID.tableAlias(environmentAlias)),
        JoinType.Left
      )
      .showPage(1, topN)
  }
}
