package com.xebialabs.deployit.support.report

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.artifact.{Artifact, DerivedArtifact}
import com.xebialabs.deployit.plugin.api.udm.{Container, Deployable, Deployed, DeployedApplication}
import com.xebialabs.deployit.repository.WorkDir
import com.xebialabs.deployit.repository.placeholders.ResolvedPlaceholderRepository
import com.xebialabs.deployit.repository.sql.CiRepository
import com.xebialabs.deployit.task.archive.TaskArchive
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.xlplatform.sugar.TempDirectorySugar
import grizzled.slf4j.Logging
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.io.FileNotFoundException
import java.lang.reflect.UndeclaredThrowableException
import scala.jdk.CollectionConverters._

@Service
class TopNSupportReportService @Autowired()(supportReportRepository: TopNSupportReportRepository,
                                            resolvedPlaceholderRepository: ResolvedPlaceholderRepository,
                                            ciRepository: CiRepository,
                                            taskArchive: TaskArchive) extends TempDirectorySugar with Logging {

  private val CI_REPOSITORY_READ_DEPTH = 1
  private val NUMBER_OF_TOP_STEPS_IN_TASK = 10
  private val NUMBER_OF_TOP_TASKS = 5

  private def getAmountOfResolvedPlaceholders(environmentId: String, deployedApplicationName: String) =
    resolvedPlaceholderRepository.countAllResolvedPlaceholdersForEnvironment(
      environmentId, Option.empty, Option.empty, Option.empty, Some(deployedApplicationName), Option.empty)

  private def stepScan(taskId: String, topN: Int = NUMBER_OF_TOP_STEPS_IN_TASK) = {
    val taskReader = taskArchive.getTask(taskId)
    val archivedTask = taskReader.fully
    val slowestSteps = archivedTask
      .getSteps
      .asScala
      .map(step => {
        val duration = if (step.getCompletionDate != null && step.getStartDate != null) {
          step.getCompletionDate.getMillis - step.getStartDate.getMillis
        }
        else {
          -1L
        }
        (step.getDescription, duration)
      })
      .sortBy(_._2)(Ordering[Long].reverse)
      .take(topN)
      .toSeq

    (archivedTask.getNrSteps, slowestSteps)
  }

  private def getDeployedApplicationCis(topApplications: Seq[TopNDeployedApplicationData],
                                        workDir: WorkDir): Seq[(TopNDeployedApplicationData, Option[DeployedApplication])] =
    topApplications match {
      case Seq() => Seq.empty
      case _ =>
        val deployedApplicationIdToDeployedApplication = ciRepository.read[DeployedApplication](
          topApplications.map(_.deployedApplicationId).asJava,
          workDir,
          CI_REPOSITORY_READ_DEPTH,
          useCache = true,
          decryptPasswords = false,
          skipNotExistingCis = true
        ).asScala
          .groupBy(_.getId)
          .view
          .mapValues(_.headOption)

        topApplications
          .map(topApplication => (
            topApplication,
            deployedApplicationIdToDeployedApplication.get(topApplication.deployedApplicationId).flatten
          ))
    }

  @SuppressWarnings(Array("IsInstanceOf"))
  private def evalArtifactsFromDeployed(artifacts: Artifacts, deployed: Deployed[_ <: Deployable, _ <: Container]): Artifacts = {
    val numberOfResolvedPlaceholders = deployed match {
      case derivedArtifact: DerivedArtifact[_] =>
        derivedArtifact.getPlaceholders.size()
      case _ => 0
    }

    deployed.getDeployable match {
      case deployableArtifact: Artifact =>
        try {
          val artifactSize = deployableArtifact.getFile.length()

          if (deployableArtifact.getProperty("scanPlaceholders"): Boolean) {
            artifacts.copy(
              scanPlaceholdersOn = artifacts.scanPlaceholdersOn.newSize(artifactSize),
              numberOfResolvedPlaceholders = artifacts.numberOfResolvedPlaceholders + numberOfResolvedPlaceholders
            )
          } else {
            artifacts.copy(
              scanPlaceholdersOff = artifacts.scanPlaceholdersOff.newSize(artifactSize),
              numberOfResolvedPlaceholders = artifacts.numberOfResolvedPlaceholders + numberOfResolvedPlaceholders
            )
          }
        } catch {
          case e: UndeclaredThrowableException if e.getUndeclaredThrowable.isInstanceOf[FileNotFoundException] => artifacts.copy(
            numberNotFoundArtifacts = artifacts.numberNotFoundArtifacts + 1,
            numberOfResolvedPlaceholders = artifacts.numberOfResolvedPlaceholders + numberOfResolvedPlaceholders
          )
          case e =>
            logger.warn("Unknown error during collecting artifact data", e)
            artifacts.copy(
              numberArtifactsErrors = artifacts.numberArtifactsErrors + 1,
              numberOfResolvedPlaceholders = artifacts.numberOfResolvedPlaceholders + numberOfResolvedPlaceholders
            )
        }
      case _ => artifacts.copy(
        numberOfResolvedPlaceholders = artifacts.numberOfResolvedPlaceholders + numberOfResolvedPlaceholders
      )
    }
  }

  private def deployedsScan(deployedApplicationCi: DeployedApplication): Artifacts = {
    val artifactType = Type.valueOf("udm.Artifact")
    deployedApplicationCi.getDeployeds
      .asScala
      .filter(deployed => deployed.getType.instanceOf(artifactType))
      .foldLeft(Artifacts())(evalArtifactsFromDeployed)
  }

  private def getTopTasks(begin: DateTime, end: DateTime,
                          deployedApplicationCi: DeployedApplication, topApplication: TopNDeployedApplicationData,
                          fetchSteps: Boolean): Seq[TopTask] = {
    val archivedTasks = supportReportRepository.topNTasks(begin, end, NUMBER_OF_TOP_TASKS, deployedApplicationCi, topApplication)
      .sortBy(_.durationMillis)
      .reverse
      .take(NUMBER_OF_TOP_TASKS)

    archivedTasks
      .map { task =>
        val (numberOfSteps, slowestStepsDurationMillis) =
          if (fetchSteps) {
            stepScan(task.taskId)
          }
          else {
            (-1, Seq())
          }
        TopTask(task.durationMillis, numberOfSteps, slowestStepsDurationMillis, task.taskId)
      }
  }

  def topNDeployedApplications(begin: DateTime, end: DateTime, topN: Int, fetchSteps: Boolean = false): Seq[TopNDeployedApplicationsReportLine] = {

    val deployedApplications = supportReportRepository.topNDeployedApplications(begin, end, topN)

    withTempDirectory { tempDir =>

      val workDir = new WorkDir(LocalFile.from(tempDir.toFile))
      getDeployedApplicationCis(deployedApplications, workDir)
        .map { case (topDeployedApplicationData, deployedApplicationCiMaybe) =>

          deployedApplicationCiMaybe.map{deployedApplicationCi =>
            val artifacts = deployedsScan(deployedApplicationCi)
            val amountOfResolvedPlaceholders =
              getAmountOfResolvedPlaceholders(topDeployedApplicationData.environmentId, deployedApplicationCi.getName)
            val topTasks = getTopTasks(begin, end, deployedApplicationCi, topDeployedApplicationData, fetchSteps)

            TopNDeployedApplicationsReportLine(
              amountOfDeployeds = topDeployedApplicationData.amountOfDeployeds,
              amountOfResolvedPlaceholders = amountOfResolvedPlaceholders,
              artifacts = artifacts,
              topTasks = topTasks,
              optimizePlan = deployedApplicationCi.isOptimizePlan,
              orchestrators = Option(deployedApplicationCi.getOrchestrator).map(_.asScala).toSeq.flatten,
              deployedApplicationId = topDeployedApplicationData.deployedApplicationId,
              environmentId = topDeployedApplicationData.environmentId
            )
          }
        }
        .flatMap(_.toSeq)
        .sortBy(_.amountOfDeployeds).reverse
    }
  }


}
