package com.xebialabs.deployit.core.upgrade.configuration

import com.xebialabs.deployit.core.upgrade.configuration.common.BaseConfigurationUpgrader
import com.xebialabs.deployit.server.api.upgrade.Version
import org.yaml.snakeyaml.{DumperOptions, Yaml}

import java.io.File
import java.nio.file.Files
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

/**
 * Configuration Upgrader for 26.1.0 to add the inMemoryThreshold property to deploy-artifact-resolver.yaml.
 * This property controls the threshold for in-memory processing of large files.
 * Files larger than this size will use streaming approach instead of loading into memory.
 */
class ArtifactResolver2610InMemoryThresholdUpgrader extends BaseConfigurationUpgrader {

  override def destinationFileName: String = "deploy-artifact-resolver.yaml"

  override def configurationFileDescription: String = "Artifact Resolver Configuration"

  val DEPLOY_ARTIFACT_IN_MEMORY_THRESHOLD = "deploy.artifact.inMemoryThreshold"
  val DEFAULT_IN_MEMORY_THRESHOLD: Long = 2147483648L

  override def upgradeVersion(): Version = Version.valueOf("central-config", "26.1.0")

  override def doUpgrade(): Boolean = Try {
    updateConfiguration()
  } match {
    case Success(_) =>
      logger.info(s"Added '$DEPLOY_ARTIFACT_IN_MEMORY_THRESHOLD' to $destinationFileName")
      true
    case Failure(exception: Exception) =>
      logger.info(s"$destinationFileName configuration cannot be updated: " + exception.getMessage)
      false
    case _ =>
      false
  }

  override protected def updateConfiguration(): Unit = {
    val configFile = new File(s"$centralConfFolder/$destinationFileName")
    if (configFile.exists()) {
      val originalMap = ConfigUtils.readYamlFile(configFile)
      
      // Check if the property already exists
      if (!hasInMemoryThreshold(originalMap)) {
        // Add the new property with default value
        updateCentralConfigurationProperties(
          Map(DEPLOY_ARTIFACT_IN_MEMORY_THRESHOLD -> Long.box(DEFAULT_IN_MEMORY_THRESHOLD))
        )
        logger.debug(s"Added $DEPLOY_ARTIFACT_IN_MEMORY_THRESHOLD with default value $DEFAULT_IN_MEMORY_THRESHOLD")
      } else {
        logger.debug(s"$DEPLOY_ARTIFACT_IN_MEMORY_THRESHOLD already exists, skipping upgrade")
      }
    }
  }

  private def hasInMemoryThreshold(map: java.util.Map[String, AnyRef]): Boolean = {
    map.asScala.get("deploy") match {
      case Some(deployMap: java.util.Map[_, _]) =>
        deployMap.asInstanceOf[java.util.Map[String, AnyRef]].asScala.get("artifact") match {
          case Some(artifactMap: java.util.Map[_, _]) =>
            artifactMap.asInstanceOf[java.util.Map[String, AnyRef]].containsKey("inMemoryThreshold")
          case _ => false
        }
      case _ => false
    }
  }

  private object ConfigUtils {
    def readYamlFile(file: File): java.util.Map[String, AnyRef] = {
      val options = new DumperOptions()
      options.setPrettyFlow(true)
      options.setDefaultFlowStyle(DumperOptions.FlowStyle.BLOCK)
      val parser = new Yaml(options)
      parser.load(Files.readString(file.toPath))
    }
  }
}
