package com.xebialabs.deployit.core.upgrade.configuration

import ai.digital.configuration.central.deploy.ServerSideProperties
import com.typesafe.config.Config
import com.xebialabs.deployit.core.upgrade.configuration.common.{BaseConfigurationUpgrader, ConfigUtils}
import com.xebialabs.deployit.server.api.upgrade.Version

import scala.util.{Failure, Success, Try}

/*
Configuration Upgrader for 10.3 for left over configurations in xl-deploy.conf to be moved to deploy-server.yaml
*/
class Deploy1030ServerConfigurationUpgrader extends BaseConfigurationUpgrader {

  protected var xlDeployConfOption: Option[Config] = None

  override def destinationFileName: String = "deploy-server.yaml"

  override def configurationFileDescription: String = "Server config"

  override def upgradeVersion(): Version = Version.valueOf("central-config", "10.3.0")

  override def doUpgrade(): Boolean = Try {
    updateConfiguration()
  } match {
    case Success(_) =>
      logger.debug(s"$configurationFileDescription configuration has been updated successfully.")
      true
    case Failure(exception: Exception) =>
      logger.debug(s"$configurationFileDescription configuration cannot be updated: " + exception.getMessage)
      false
    case _ =>
      false
  }

  override protected def updateConfiguration(): Unit = {
    xlDeployConfOption = ConfigUtils.loadXlDeployConf(xlDeployConfFile)
    xlDeployConfOption.map(conf => {
      Try {
        // Read from xl-deploy and create map of configs
        val confToUpdate = getConfToUpdate(conf)
        updateCentralConfigurationProperties(confToUpdate)
        // clean up
        excludeXlDeployConfSections(List(XL_REST_PATH, XL_EXPORT_CIS_PATH))
        logger.debug(s"Moved $confToUpdate from xl-deploy.conf to deploy-server.yaml")
      } match {
        case Failure(e) =>
          logger.error(s"Failure in moving configuration form xl-deploy.conf to : $destinationFile", e)
          throw e
        case _ =>
      }
    })
  }

  /**
   * Read from xl-deploy.conf and create map for the specific set of confs to be updated to the destination file
   * @param deployConf - config from conf file
   * @return map of subset config from server config
   */
  def getConfToUpdate(deployConf: Config): Map[String, AnyRef] = {
    var confToUpdate = Map[String, AnyRef]()
    if (deployConf.hasPath(XL_REST_PATH)) {
      confToUpdate  +=  (SERVER_REST_PATH_API_MAX_PAGE_SIZE ->
        ConfigUtils.getValueOrDefault(deployConf, XL_REST_PATH_API_MAX_PAGE_SIZE, Int.box(ServerSideProperties.DEFAULT_MAX_PAGE_SIZE)))
    }
    if (deployConf.hasPath(XL_EXPORT_CIS_PATH)) {
      confToUpdate += (SERVER_EXPORT_CIS_PATH_EXPORT_DIR ->
        ConfigUtils.getValueOrDefault(deployConf, XL_EXPORT_CIS_PATH_EXPORT_DIR, ServerSideProperties.DEFAULT_EXPORT_DIR))
    }
    if (deployConf.hasPath(XL_EXPORT_CIS_PATH_IMPORT_DIR)) {
      confToUpdate += (SERVER_EXPORT_CIS_PATH_IMPORT_DIR ->
        ConfigUtils.getValueOrDefault(deployConf, XL_EXPORT_CIS_PATH_IMPORT_DIR, ServerSideProperties.DEFAULT_IMPORT_WORK_DIR))
    }
    confToUpdate
  }

  val XL_REST_PATH = "xl.rest"
  val XL_REST_PATH_API_MAX_PAGE_SIZE = "xl.rest.api.maxPageSize"
  val XL_EXPORT_CIS_PATH = "xl.export-cis"
  val XL_EXPORT_CIS_PATH_EXPORT_DIR = "xl.export-cis.export-dir"
  val XL_EXPORT_CIS_PATH_IMPORT_DIR = "xl.export-cis.import-work-dir"
  val SERVER_REST_PATH_API_MAX_PAGE_SIZE = "deploy.server.rest.api.maxPageSize"
  val SERVER_EXPORT_CIS_PATH_EXPORT_DIR = "deploy.server.export-cis.export-dir"
  val SERVER_EXPORT_CIS_PATH_IMPORT_DIR = "deploy.server.export-cis.import-work-dir"
}
