package com.xebialabs.deployit.core.upgrade.configuration.common

import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.deployit.util.PropertyUtil
import com.xebialabs.xlplatform.config.ConfigLoader

import java.io.{File}
import java.util
import java.util.Properties
import scala.util.{Failure, Try}

trait ComposedConfUpgrader[T] extends DeployConfUpgrader {

  type ConfigurationTransformerFn = (T, String) => Unit

  type AddIfExistsFn = (List[String], ConfigurationTransformerFn) => Unit

  protected var xlDeployConfOption: Option[Config] = None

  protected def xlDeployConf: Config = xlDeployConfOption
    .getOrElse(ConfigLoader.loadWithDynamic(ConfigFactory.empty()))

  override protected def updateConfiguration(): Unit = {
    loadXlDeployConf()

    val consolidatedConfig = Function.chain(Seq(
      readDeployItConf(_),
      readXlDeployConf(_),
      readCentralConfigYamlFile(_)
    ))(getInitialConfig)

    createYamlConfiguration(yamlPrefix, consolidatedConfig)
    cleanUp()
  }

  private def cleanUp(): Unit = {
    excludeXlDeployConfSections(listXlDeployConfPrefixes)
    removeInsourcedYamlFiles()
    removeEntriesInDeployitConf()
  }

  private def removeInsourcedYamlFiles(): Unit = {
    listInsourcedYamlFiles.foreach { listInsourcedYamlFile =>
      if (listInsourcedYamlFile.exists()) {
        listInsourcedYamlFile.delete()
      }
    }
  }

  private def removeEntriesInDeployitConf(): Unit = {
    if (deployitConfFile.exists()) {
      val properties = PropertyUtil.readPropertiesFile(deployitConfFile)
      listDeployitConfKeyNames.foreach(properties.remove)
      PropertyUtil.storePropertiesFile(deployitConfFile, properties)
    }
  }

  protected def createYamlConfiguration(yamlPrefix: String, config: T): Unit = {
    Try {
      val pairs =
      Seq((yamlPrefix, ConfigUtils.configAsMap(config))) ++ additionalYamlConfigs().map {
        case (key, value) => (key, ConfigUtils.configAsMap(value))
      }
      storeCentralConfigurationProperties(pairs)
    }
    match {
      case Failure(e) =>
        logger.error(s"Failure during migration in: $destinationFile", e)
        throw e
      case _ =>
    }
  }

  protected def additionalYamlConfigs(): Map[String, AnyRef] = Map()

  protected def readXlDeployConf(config: T): T =
    xlDeployConfOption
      .map(processXlDeployConf(_, config))
      .getOrElse(config)

  protected def loadXlDeployConf(): Unit = {
    xlDeployConfOption = ConfigUtils.loadXlDeployConf(xlDeployConfFile)
  }

  protected def readDeployItConf(config: T): T = {
    if (deployitConfFile.exists()) {
      val properties = PropertyUtil.readPropertiesFile(deployitConfFile)
      return processDeployItConf(properties, config)
    }
    config
  }

  protected def readCentralConfigYamlFile(config: T): T = {
    listInsourcedYamlFiles.foreach { listInsourcedYamlFile =>
      if (listInsourcedYamlFile.exists()) {
        val map = ConfigUtils.readYamlFile(listInsourcedYamlFile)
        def addIfExists(keyPath: List[String], fn: ConfigurationTransformerFn): Unit = {
          def hasKey: Boolean = try {
            getValue
            true
          } catch {
            case _: Exception => false
          }

          def getValue: String = {
            val v = keyPath.dropRight(1).foldLeft(map) { (acc, key) =>
              acc.get(key).asInstanceOf[util.Map[String, AnyRef]]
            }
            v.get(keyPath.lastOption.orNull).toString
          }

          if (hasKey) {
            fn(config, getValue)
          }
        }

        processCentralConfigYamlFile(map, config, addIfExists)
      }
    }

    config
  }

  protected def processXlDeployConf(xlDeployConf: Config, config: T): T

  protected def processDeployItConf(properties: Properties, config: T): T

  protected def processCentralConfigYamlFile(map: util.Map[String, AnyRef], config: T, addIfExistsFn: AddIfExistsFn): T

  protected def getInitialConfig: T

  protected def listXlDeployConfPrefixes: List[String]

  protected def yamlPrefix: String

  protected def listInsourcedYamlFiles: List[File]

  protected def listDeployitConfKeyNames: List[String]

  override protected def xlConfigUpdate(confFile: Config): Unit = {}

}
