package com.xebialabs.deployit.repository.sql.artifacts.mover

import java.io.{File, InputStream}
import java.sql.ResultSet
import java.util

import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.repository.sql.artifacts._
import com.xebialabs.deployit.repository.sql.base.{CiPKType, asCiPKType}
import org.springframework.jdbc.core.{ResultSetExtractor, SingleColumnRowMapper}

import scala.jdk.CollectionConverters._

trait ArtifactMoverRepository {
  def artifactDataRepository: ArtifactDataRepository
  def isEmpty: Boolean = artifactDataRepository.countArtifacts == 0
}

class FileArtifactMoverRepository(private val fileArtifactRepository: FileArtifactDataRepository)
  extends FileArtifactMoverQueries(fileArtifactRepository.schemaInfo) with ArtifactMoverRepository {
  override val artifactDataRepository: ArtifactDataRepository = fileArtifactRepository

  private val jdbcTemplate = fileArtifactRepository.jdbcTemplate

  def getLastID(ignore: util.Set[Number]): Option[Number] = {
    val selectBuilder = new SelectBuilder(FileArtifactSchema.tableName)(fileArtifactRepository.schemaInfo)
      .select(SqlFunction.max(FileArtifactSchema.ID))
    if (!ignore.isEmpty) {
      selectBuilder
        .where(SqlCondition.not(SqlCondition.in(FileArtifactSchema.ID, ignore.asScala)))
    }
    Option(jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new SingleColumnRowMapper[Number]()).get(0))
  }

  def getUsages(id: Number): util.List[CiPKType] =
    jdbcTemplate.query(SELECT_USAGES_BY_ID, (rs: ResultSet, _: Int) => {
      asCiPKType(rs.getInt(FileArtifactUsageSchema.ci_id.name))
    }, id)

  def remove(id: Number): Unit = {
    jdbcTemplate.update(DELETE_USAGES_BY_ID, id)
    fileArtifactRepository.remove(id)
  }

  def insertUsages(id: Int, usages: util.List[CiPKType]): Unit = {
    usages.forEach(fileArtifactRepository.insertUsage(id, _))
  }

  def getLocation(id: Number): File =
    fileArtifactRepository.getLocation(id)

  def storeToTemp(inputStream: InputStream): File =
    fileArtifactRepository.storeToTemp(inputStream)

  def moveTempToChecksumBasedFile(tempFile: File, checksum: String): String = {
    val path = FileArtifactDataRepositoryImpl.pathFromChecksum(checksum)
    val file = fileArtifactRepository.fileFromRelativePath(path)
    FileArtifactDataRepositoryImpl.moveTempFileToArtifactFile(tempFile, file)
    path
  }

  def insert(location: String): Int =
    fileArtifactRepository.insert(location)
}

class FileArtifactMoverQueries(override val schemaInfo: SchemaInfo) extends Queries {
  import com.xebialabs.deployit.repository.sql.artifacts.FileArtifactUsageSchema.{artifact_id, ci_id, tableName => usagesTableName}

  val SELECT_USAGES_BY_ID = sqlb"select $ci_id from $usagesTableName where $artifact_id = ?"
  val DELETE_USAGES_BY_ID = sqlb"delete from $usagesTableName where $artifact_id = ?"
}

class DbArtifactMoverRepository(private val dbArtifactDataRepository: DbArtifactDataRepository, private val artifactRepository: DbArtifactRepository)
  extends DbArtifactMoverQueries(dbArtifactDataRepository.schemaInfo) with ArtifactMoverRepository {
  override val artifactDataRepository: ArtifactDataRepository = dbArtifactDataRepository

  private val jdbcTemplate = dbArtifactDataRepository.jdbcTemplate

  def insertUsages(checksum: String, usages: util.List[CiPKType]): Unit = {
    usages.forEach(artifactDataRepository.insert(_, checksum))
  }

  def getLastChecksum(ignore: util.Set[String]): Option[String] = {
    val selectBuilder = new SelectBuilder(DbArtifactSchema.tableName)(dbArtifactDataRepository.schemaInfo)
      .select(SqlFunction.max(DbArtifactSchema.ID))
    if (!ignore.isEmpty) {
      selectBuilder
        .where(SqlCondition.not(SqlCondition.in(DbArtifactSchema.ID, ignore.asScala)))
    }
    Option(jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new SingleColumnRowMapper[String]()).get(0))
  }

  def withInputStream[T](id: String)(block: InputStream => T): T = {
    val extractor: ResultSetExtractor[T] = (rs: ResultSet) => if (rs.next()) block(rs.getBinaryStream(1)) else throw new IllegalStateException("No data.")
    jdbcTemplate.query(SELECT_ARTIFACT_BY_ID, extractor, id)
  }

  def getUsages(id: String): util.List[CiPKType] =
    jdbcTemplate.query(SELECT_USAGES_BY_ID, (rs: ResultSet, _: Int) => {
      asCiPKType(rs.getInt(FileArtifactUsageSchema.ci_id.name))
    }, id)

  def remove(id: String): Unit = {
    jdbcTemplate.update(DELETE_USAGES_BY_ID, id)
    artifactRepository.removeArtifact(id)
  }

  def insert(id: String, inputStream: InputStream): Int =
    artifactRepository.insertArtifact(id, inputStream)
}

class DbArtifactMoverQueries(override val schemaInfo: SchemaInfo) extends Queries {

  import com.xebialabs.deployit.repository.sql.artifacts.DbArtifactSchema.{ID, data, tableName => artifactsTableName}
  import com.xebialabs.deployit.repository.sql.artifacts.DbArtifactUsageSchema.{artifact_id, ci_id, tableName => usagesTableName}

  val SELECT_ARTIFACT_BY_ID = sqlb"select $data from $artifactsTableName where $ID = ?"

  val SELECT_USAGES_BY_ID = sqlb"select $ci_id from $usagesTableName where $artifact_id = ?"
  val DELETE_USAGES_BY_ID = sqlb"delete from $usagesTableName where $artifact_id = ?"
}
