/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */

package com.xebialabs.deployit.plugin.stitch.service.engine.processor.handler;

import com.fasterxml.jackson.databind.JsonNode;
import com.xebialabs.deployit.plugin.stitch.model.StitchProcessor;
import com.xebialabs.deployit.plugin.stitch.service.engine.context.DeploymentContext;
import com.xebialabs.deployit.plugin.stitch.service.engine.index.dto.StitchProcessorDto;
import com.xebialabs.deployit.plugin.stitch.service.engine.processor.InvalidProcessorException;
import com.xebialabs.xlplatform.documentation.PublicApi;

import java.util.Map;

/**
 * Processor handler that can handle and apply stitch processor,
 * in other words it is responsible for stitch transformation.
 */
@PublicApi
public interface ProcessorHandler {

  /**
   * Returns type that can be handled by this handler.
   * @return type that can be handled by handler.
   */
  String type();

  /**
   * Returns file parameter name for external file.
   * @return parameter name that corresponds to one that can be saved externally.
   */
  String fileParameterName();

  /**
   * This method is responsible for stitch transformation.
   * It should do whatever is expected from processor to transform input.
   *
   * @param input             to handler as {@link JsonNode}
   * @param processor         that is being applied
   * @param deploymentContext context for deployment associated with stitch transformation
   * @param params            parameters given to processor
   * @return transformed input with applied processor as {@link JsonNode}.
   * @throws Exception when handling of a processor failed
   */
  JsonNode handle(
          JsonNode input,
          StitchProcessor processor,
          DeploymentContext deploymentContext,
          Map<String, Object> params
          ) throws Exception;

  /**
   * Method for processor validation during parsing/validation phase.
   *
   * @param processor to be validated
   * @throws InvalidProcessorException when processor can't pass validation
   */
  void validate(StitchProcessorDto processor) throws InvalidProcessorException;
}
